const cron = require("node-cron");
const {
  sendDailySalesSummary,
  createProductExpiryNotification,
  createOutOfStockNotification,
  createProductSoonExpiryNotification,
  sendSalaryReminderNotification, // You'll need to implement this in your notificationController
} = require("../controllers/notificationController");
const UserPreferences = require("../models/userPreferencesModel");
const ActivityTracker = require("../models/activityTrackerModel");
const DeletionFrequency = require("../models/deletionFrequencyModel");
const Product = require("../models/productModel");
const Outofstocklimit = require("../models/outofstocklimitModel");
const LoginHistory = require("../models/loginHistoryModel");

// ====== Constants and Initial State ======
const DELETION_FREQUENCIES = ["hourly", "weekly", "monthly", "yearly"];
let deletionFrequency = "monthly"; // Default to monthly
let currentTask = null;

// ====== Activity Tracker Deletion Functions ======
const deleteActivities = async () => {
  try {
    await ActivityTracker.deleteMany({});
  } catch (error) {
    console.error("Error deleting activities:", error);
  }
};

const scheduleDeletion = () => {
  if (currentTask) {
    currentTask.stop(); // Stop the existing task
  }

  const cronExpressions = {
    hourly: "*/5 * * * *", // Every hour
    weekly: "0 0 * * 0", // Every Sunday at midnight
    monthly: "0 0 1 * *", // First day of every month at midnight
    yearly: "0 0 1 1 *", // January 1st at midnight
  };

  currentTask = cron.schedule(
    cronExpressions[deletionFrequency] || "0 0 1 * *",
    deleteActivities
  );
};

const initializeDeletionFrequency = async () => {
  try {
    const result = await DeletionFrequency.findOne({
      slug: "deletionfrequency",
    });

    if (result) {
      deletionFrequency = result.name;
    }
    scheduleDeletion(); // Schedule the initial deletion task
  } catch (error) {
    console.error("Error initializing auto script:", error);
  }
};

const updateDeletionSettings = (frequency) => {
  if (!DELETION_FREQUENCIES.includes(frequency)) {
    throw new Error("Invalid frequency");
  }
  deletionFrequency = frequency;
  scheduleDeletion(); // Reschedule the task with the new frequency
};

// ====== Product Expiration Functions ======
const checkExpiredProducts = async (req) => {
  try {
    // 1. Deactivate expired products
    const expiredProducts = await Product.find({
      expireDate: { $lte: new Date() },
      active: { $ne: false },
    }).select("+active");

    await Promise.all(
      expiredProducts.map(async (product) => {
        product.active = false;
        await product.save();

        // Notify about product expiration
        if (product.user) {
          await createProductExpiryNotification(product.user, {
            ...product.toObject(),
            name: `${product.name} (EXPIRED)`,
          });
        }
      })
    );
  } catch (error) {
    console.error("Error checking product expirations:", error);
    throw error;
  }
};

const checkAbouToExpiredProducts = async (req) => {
  var date = new Date();
  var date10 = new Date(date.getTime());
  date10.setDate(date10.getDate() + 10);
  try {
    const expiredProducts = await Product.find({
      expireDate: { $lte: new Date(date10), $gte: new Date() },
      hideExpireDate: { $ne: false },
    }).select("+active");

    await Promise.all(
      expiredProducts.map(async (product) => {
        product.active = false;
        await product.save();
        if (product.user) {
          await createProductSoonExpiryNotification(product.user, {
            ...product.toObject(),
            name: `${product.name} (EXPIRED)`,
          });
        }
      })
    );
  } catch (error) {
    console.error("Error checking product expirations:", error);
    throw error;
  }
};

const checkOutOfStockProducts = async (req) => {
  try {
    const limit = await Outofstocklimit.find({});
    const stocklimit = limit[0].stocklimit;
    const outOfStockProducts = await Product.find({
      quantity: { $lte: stocklimit },
      active: true,
    }).select("+active");

    await Promise.all(
      outOfStockProducts.map(async (product) => {
        if (product.user) {
          await createOutOfStockNotification(product.user, product);
        }
      })
    );
  } catch (error) {
    console.error("Error checking out-of-stock products:", error);
  }
};

const setupProductCron = () => {
  cron.schedule("0 12 * * *", async () => {
    try {
      await checkExpiredProducts();
      await checkAbouToExpiredProducts();
      await checkOutOfStockProducts();
    } catch (error) {
      console.error("Error in product expiry cron job:", error);
    }
  });
};

// ====== Daily Summary Notification Functions ======
const setupDailySummaryCron = () => {
  cron.schedule("* * * * *", async () => {
    try {
      const now = new Date();
      const currentHour = String(now.getHours()).padStart(2, "0");
      const currentMinute = String(now.getMinutes()).padStart(2, "0");
      const currentTime = `${currentHour}:${currentMinute}`;

      const users = await UserPreferences.find({
        "notificationPreferences.dailySummary": true,
        dailySummaryTime: currentTime,
      }).populate("user");

      for (const prefs of users) {
        if (prefs.notificationPreferences.pushEnabled) {
          try {
            await sendDailySalesSummary(prefs.user._id);
          } catch (error) {
            console.error(`Failed for ${prefs.user._id}:`, error.message);
          }
        }
      }
    } catch (error) {
      console.error("Cron job failed:", error);
    }
  });
};
// ====== Daily Summary Notification Functions ======
const setupLoginHistoryDeletionCron = () => {
  cron.schedule("* * * * *", async () => {
    try {
      const result = await LoginHistory.deleteMany({});
      console.log(`Deleted ${result.deletedCount} login history records`);
    } catch (error) {
      console.error("Error deleting login history:", error);
    }
  });
};

// ====== Salary Reminder Functions ======
const setupSalaryReminderCron = () => {
  // Runs every minute for testing purposes
  cron.schedule("* * * * *", async () => {
    try {
      const now = new Date();
      const currentDay = now.getDate();
      const currentHour = now.getHours();
      const currentMinute = now.getMinutes();
      // Find users with reminders enabled for today
      const usersToNotify = await UserPreferences.find({
        "notificationPreferences.salaryReminder": true,
        salaryReminderDate: currentDay,
      }).populate("user");
      for (const pref of usersToNotify) {
        const [reminderHour, reminderMinute] = pref.salaryReminderTime
          .split(":")
          .map(Number);

        // Check if current time matches reminder time
        if (currentHour === reminderHour && currentMinute === reminderMinute) {
          // console.log(`Sending salary reminder to ${pref.providedEmail}`);
          await sendSalaryReminderNotification(pref.user._id, {
            reminderDate: pref.salaryReminderDate,
            reminderTime: pref.salaryReminderTime,
            providedEmail: pref.providedEmail,
          });
        }
      }
    } catch (error) {
      console.error("Error in salary reminder cron job:", error);
    }
  });
};

// ====== Initialization Function ======
const initializeCronJobs = () => {
  setupProductCron();
  sendSalaryReminders();
  setupDailySummaryCron();
  setupSalaryReminderCron();
  initializeDeletionFrequency();
  setupLoginHistoryDeletionCron();
};

// ====== Module Exports ======
module.exports = {
  initializeCronJobs,
  setupProductCron,
  setupSalaryReminderCron,
  setupDailySummaryCron,
  updateDeletionSettings,
  initializeDeletionFrequency,
  setupLoginHistoryDeletionCron,
};
