const {
  sendDailySalesSummary,
} = require("../controllers/notificationController");
const UserPreferences = require("../models/userPreferencesModel");
const ActivityTracker = require("../models/activityTrackerModel");
const DeletionFrequency = require("../models/deletionFrequencyModel");
const Product = require("../models/productModel");
const cron = require("node-cron");

// ====== Deletion Frequency for Activity Tracker ======
let deletionFrequency = "monthly"; // Default to monthly
let currentTask = null;

const deleteActivities = async () => {
  try {
    await ActivityTracker.deleteMany({});
    // console.log("Activities deleted successfully");
  } catch (error) {
    console.error("Error deleting activities:", error);
  }
};

const scheduleDeletion = () => {
  if (currentTask) {
    currentTask.stop(); // Stop the existing task
  }

  let cronExpression;
  switch (deletionFrequency) {
    case "hourly":
      cronExpression = "*/5 * * * *"; // Every hour
      break;
    case "weekly":
      cronExpression = "0 0 * * 0"; // Every Sunday at midnight
      break;
    case "monthly":
      cronExpression = "0 0 1 * *"; // The first day of every month at midnight
      break;
    case "yearly":
      cronExpression = "0 0 1 1 *"; // January 1st at midnight
      break;
    default:
      cronExpression = "0 0 1 * *"; // Default to monthly
      break;
  }

  currentTask = cron.schedule(cronExpression, deleteActivities);
};

const initializeDeletionFrequency = async () => {
  try {
    const result = await DeletionFrequency.findOne({
      slug: "deletionfrequency",
    });

    if (result) {
      deletionFrequency = result.name;
    }
    scheduleDeletion(); // Schedule the initial deletion task
  } catch (error) {
    console.error("Error initializing auto script:", error);
  }
};

const updateDeletionSettings = (frequency) => {
  if (!["hourly", "weekly", "monthly", "yearly"].includes(frequency)) {
    throw new Error("Invalid frequency");
  }
  deletionFrequency = frequency;
  scheduleDeletion(); // Reschedule the task with the new frequency
};

// ====== Expired Product Check (Runs Every 8 Minutes) ======
const checkExpiredProducts = async () => {
  try {
    const products = await Product.find({
      expireDate: { $lte: new Date() }, // Products with an expire date <= current date
      active: { $ne: false }, // Only products where hideExpireDate is not false
    }).select("+active");

    // Update each expired product
    await Promise.all(
      products.map(async (product) => {
        product.active = false;
        // product.hideExpireDate = false;
        await product.save(); // Save the updated product
      })
    );

    // console.log(
    //   `${products.length} products updated to inactive due to expiration.`
    // );
  } catch (error) {
    console.error("Error checking expired products:", error);
  }
};

const startProductExpirationCheck = () => {
  setInterval(checkExpiredProducts, 24 * 60 * 60 * 1000); // 24 hours
};

const setupDailySummaryCron = () => {
  // Run every minute for testing (change to "*/5 * * * *" for every 5 minutes in production)
  cron.schedule("*/5 * * * *", async () => {
    try {
      const now = new Date();
      const currentHour = String(now.getHours()).padStart(2, "0");
      const currentMinute = String(now.getMinutes()).padStart(2, "0");
      const currentTime = `${currentHour}:${currentMinute}`;

      // 1. Find all users who want summaries at this exact minute
      const users = await UserPreferences.find({
        "notificationPreferences.dailySummary": true,
        dailySummaryTime: currentTime,
      }).populate("user");

      // 2. Debug output if no users found
      if (users.length === 0) {
        const sampleUser = await UserPreferences.findOne({
          "notificationPreferences.dailySummary": true,
        });
        // console.log(
        //   `No users for ${currentTime}. Sample user time: ${sampleUser?.dailySummaryTime}`
        // );
      }

      // 3. Process notifications
      for (const prefs of users) {
        if (prefs.notificationPreferences.pushEnabled) {
          try {
            await sendDailySalesSummary(prefs.user._id);
          } catch (error) {
            console.error(`Failed for ${prefs.user._id}:`, error.message);
          }
        }
      }
    } catch (error) {
      console.error("Cron job failed:", error);
    }
  });
};

// Call this function when your app starts
const initializeScripts = () => {
  startProductExpirationCheck(); // Start checking for expired products
};

// ====== Exports ======
module.exports = {
  initializeDeletionFrequency,
  updateDeletionSettings,
  initializeScripts,
  setupDailySummaryCron,
};

// Start the interval to check for expired products every 8 minutes (480,000 milliseconds)
// const startProductExpirationCheck = () => {
//   setInterval(checkExpiredProducts, 50 * 60 * 1000); // 50 minutes
// };
