const Activity = require("../models/activityTrackerModel");
const catchAsync = require("../utils/catchAsync");

exports.getAllActivityTracker = catchAsync(async (req, res, next) => {
  try {
    const { activityType = selectedActivity, limit = 12, page = 1 } = req.query;
    const matchStage = activityType ? { $match: { activityType } } : {};
    const skip = (parseInt(page) - 1) * parseInt(limit);

    const activityTracker = await Activity.aggregate([
      matchStage,
      {
        $group: {
          _id: "$activityType",
          count: { $sum: 1 },
          activities: { $push: "$$ROOT" },
        },
      },
      {
        $project: {
          _id: 1,
          count: 1,
          activities: 1,
        },
      },
      { $unwind: "$activities" },
      { $sort: { "activities.createdAt": -1 } },
      { $skip: skip },
      { $limit: parseInt(limit) },
      {
        $group: {
          _id: "$id",
          count: { $first: "$count" },
          activities: { $push: "$activities" },
        },
      },
    ]);

    // Populate references with valid ObjectIds only
    for (const activity of activityTracker) {
      await Activity.populate(activity.activities, [
        { path: "user", model: "User", select: "name" },
        { path: "details.product", model: "Product", select: "name quantity" },
      ]);
    }

    res.json({
      success: true,
      data: activityTracker[0] || {
        _id: activityType,
        count: 0,
        activities: [],
      },
    });
  } catch (error) {
    if (error.kind === "ObjectId") {
      res.status(400).json({
        success: false,
        message: "Resources not found with this id..Invalid _id",
      });
    } else {
      next(error); // Passes unexpected errors to the global error handler
    }
  }
});
