const User = require("../models/userModel");
const Activity = require("../models/activityTrackerModel");
const loginHistoryController = require("./loginHistoryController");
const catchAsync = require("../utils/catchAsync");
const { hashPassword } = require("../helpers/auth");
const AppError = require("../utils/appError");
const bcrypt = require("bcrypt");
const fs = require("fs");
const { generateTokens } = require("../utils/token");
const jwt = require("jsonwebtoken");

exports.login = async (req, res) => {
  const { identifier, password } = req.body; // identifier can be either username or email

  try {
    // Regular expression to check if the identifier is an email
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    const isEmail = emailRegex.test(identifier);

    // Build the query based on whether the identifier is an email or username
    const query = isEmail
      ? { email: identifier, active: { $ne: false } }
      : {
          $or: [{ username: identifier }, { email: identifier }],
          active: { $ne: false },
        };

    const user = await User.findOne(query)
      .populate({
        path: "role",
        model: "Role",
        select: "name",
      })
      .populate("permissions")
      .populate({
        path: "assignedRoutes",
        model: "Route",
        select: "path name",
      })
      .populate({
        path: "menus",
        model: "Menu",
        populate: {
          path: "routes",
          model: "Route",
          select: "path name",
        },
      })
      .select("+password +active ");

    if (!user) {
      // Record failed login attempt (user not found)
      // await loginHistoryController.recordLogin({ _id: null }, req, "failed");
      return res.status(400).json({ message: "User not found." });
    }

    const isPasswordMatch = await bcrypt.compare(password, user.password);
    if (!isPasswordMatch) {
      // Record failed login attempt (wrong password)
      await loginHistoryController.recordLogin(user, req, "failed");
      return res.status(404).json({
        status: false,
        message: "Invalid credentials",
      });
    }

    // Generate combinedRoutes
    const combinedItems = [];
    const routeSet = new Set(); // To track added routes

    // Add assignedRoutes first
    user.assignedRoutes.forEach((route) => {
      if (!routeSet.has(route._id.toString())) {
        combinedItems.push({
          type: "route",
          _id: route._id,
          path: route.path,
          name: route.name,
        });
        routeSet.add(route._id.toString());
      }
    });

    // Add menus and their submenus
    user.menus.forEach((menu) => {
      combinedItems.push({
        type: "menu",
        _id: menu._id,
        path: menu.path,
        name: menu.name,
      });
      menu.routes.forEach((route) => {
        if (!routeSet.has(route._id.toString())) {
          combinedItems.push({
            type: "submenu",
            _id: route._id,
            path: route.path,
            name: route.name,
            parentMenu: menu._id,
          });
          routeSet.add(route._id.toString());
        }
      });
    });

    // Update user document with combinedRoutes if it doesn't already exist
    if (user.combinedRoutes.length === 0) {
      user.combinedRoutes = combinedItems;
      await user.save();
    }

    // Generate tokens
    const { accessToken, refreshToken } = generateTokens(user);
    res.cookie("refreshToken", refreshToken, {
      httpOnly: true,
      secure: process.env.NODE_ENV === "production",
      sameSite: "lax", // For localhost
      maxAge: 15 * 24 * 60 * 60 * 1000, // 15 days
    });

    // Save refreshToken to user's document
    user.refreshToken = refreshToken;
    await user.save();

    // Record successful login
    await loginHistoryController.recordLogin(user, req);

    const activity = new Activity({
      user: user._id,
      activityType: "login",
      details: {
        ...user._doc,
      },
    });
    await activity.save();

    // Send response
    res.status(200).json({
      status: true,
      message: "Login Success",
      accessToken,
      user: {
        ...user._doc,
        combinedRoutes: combinedItems,
      },
    });
  } catch (err) {
    console.error("Login error:", err);
    res.status(500).json({ error: "Internal Server Error" });
  }
};

exports.loginOld = async (req, res) => {
  const { identifier, password } = req.body; // identifier can be either username or email

  try {
    // Regular expression to check if the identifier is an email
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    const isEmail = emailRegex.test(identifier);

    // Build the query based on whether the identifier is an email or username
    const query = isEmail
      ? { email: identifier, active: { $ne: false } }
      : {
          $or: [{ username: identifier }, { email: identifier }],
          active: { $ne: false },
        };

    const user = await User.findOne(query)
      .populate({
        path: "role",
        model: "Role",
        select: "name",
      })
      .populate("permissions")
      .populate({
        path: "assignedRoutes",
        model: "Route",
        select: "path name",
      })
      .populate({
        path: "menus",
        model: "Menu",
        populate: {
          path: "routes",
          model: "Route",
          select: "path name",
        },
      })
      .select("+password +active ");

    if (!user) {
      return res.status(400).json({ message: "User not found." });
    }

    const isPasswordMatch = await bcrypt.compare(password, user.password);
    if (!isPasswordMatch) {
      return res.status(404).json({
        status: false,
        message: "Invalid credentials",
      });
    }

    // Generate combinedRoutes
    const combinedItems = [];
    const routeSet = new Set(); // To track added routes

    // Add assignedRoutes first
    user.assignedRoutes.forEach((route) => {
      if (!routeSet.has(route._id.toString())) {
        combinedItems.push({
          type: "route",
          _id: route._id,
          path: route.path,
          name: route.name,
        });
        routeSet.add(route._id.toString());
      }
    });

    // Add menus and their submenus
    user.menus.forEach((menu) => {
      combinedItems.push({
        type: "menu",
        _id: menu._id,
        path: menu.path,
        name: menu.name,
      });
      menu.routes.forEach((route) => {
        if (!routeSet.has(route._id.toString())) {
          combinedItems.push({
            type: "submenu",
            _id: route._id,
            path: route.path,
            name: route.name,
            parentMenu: menu._id,
          });
          routeSet.add(route._id.toString());
        }
      });
    });

    // Update user document with combinedRoutes if it doesn't already exist
    if (user.combinedRoutes.length === 0) {
      user.combinedRoutes = combinedItems;
      await user.save();
    }

    // Generate tokens
    const { accessToken, refreshToken } = generateTokens(user);
    res.cookie("refreshToken", refreshToken, {
      httpOnly: true,
      secure: process.env.NODE_ENV === "production",
      sameSite: "lax", // For localhost
      maxAge: 15 * 24 * 60 * 60 * 1000, // 15 days
    });

    // Save refreshToken to user's document
    user.refreshToken = refreshToken;
    await user.save();

    const activity = new Activity({
      user: user._id,
      activityType: "login",
      details: {
        ...user._doc,
      },
    });
    await activity.save();

    // Send response
    res.status(200).json({
      status: true,
      message: "Login Success",
      accessToken,
      user: {
        ...user._doc,
        combinedRoutes: combinedItems,
      },
    });
  } catch (err) {
    console.error("Login error:", err);
    res.status(500).json({ error: "Internal Server Error" });
  }
};

exports.readUser = async (req, res) => {
  try {
    const user = await User.findById(req.user.id)
      .select("-password +active")
      .exec();
    return res.json(user);
  } catch (err) {
    console.log(err);
  }
};

exports.updateProfile = catchAsync(async (req, res, next) => {
  const { name, email, contactNum, password, username } = req.body;
  // Retrieve the uploaded image filename from req.file if available

  const uploadedFilename = req.file ? req.file.filename : "";
  // return;
  const preuser = await User.findById({ _id: req.user.id })
    .select("+active +password")
    .populate({
      path: "role",
      model: "Role",
      select: "name",
    })
    .populate({
      path: "assignedRoutes",
      model: "Route",
      select: "path name",
    })
    .populate({
      path: "menus",
      model: "Menu",
      populate: {
        path: "routes",
        model: "Route",
        select: "path name",
      },
    });

  if (
    uploadedFilename &&
    preuser?.profileimage &&
    preuser?.profileimage !== "default.jpg"
  ) {
    const filePath = `../public/img/${preuser?.profileimage}`;
    fs.unlink(filePath, (err) => {
      if (err) {
        console.log(err);
      }
    });
  }

  // check password length
  if (password && password.length < 6) {
    return next(new AppError("Password should be 6 characters long", 500));
  }

  const exist = await User.findOne({ email });

  if (exist && exist._id.toString() !== preuser._id.toString()) {
    return next(new AppError("Email is taken", 401));
  }

  const hashedPassword = password ? await hashPassword(password) : undefined;

  const user = await User.findByIdAndUpdate(
    req.user.id,
    {
      username: username || preuser.username,
      // username: name === exist?.name ? exist?.username : slugify(name),
      name: name || preuser.name,
      email: email || preuser.email,
      contactNum: contactNum || preuser.contactNum,
      password: hashedPassword || preuser.password,
      profileimage: req.file?.filename || preuser.profileimage,
      generatedPassword: " ",
    },
    { new: true }
  )
    .populate({
      path: "role",
      model: "Role",
      select: "name",
    })
    .populate({
      path: "assignedRoutes",
      model: "Route",
      select: "path name",
    })
    .populate({
      path: "menus",
      model: "Menu",
      populate: {
        path: "routes",
        model: "Route",
        select: "path name",
      },
    });

  const { accessToken, refreshToken } = generateTokens(user);
  res.cookie("refreshToken", refreshToken, {
    httpOnly: true,
    secure: process.env.NODE_ENV === "production", // Use secure cookies in production
    sameSite: "strict", // Prevent CSRF attacks
    maxAge: 15 * 24 * 60 * 60 * 1000, // 15 days
  });
  const { password: userPassword, ...userData } = user._doc;
  res.json({
    status: true,
    message: "Update Success",
    accessToken,
    user: userData,
  });
});

exports.signOut = async (req, res, next) => {
  try {
    res.cookie("refreshToken", "", {
      httpOnly: true,
      expires: new Date(0),
    });
    res.status(200).json({ message: "Logged out success" });
  } catch (error) {
    next(error);
  }
};

exports.refreshToken = async (req, res) => {
  const { refreshToken } = req.cookies;
  if (!refreshToken) {
    return res.status(403).json({ message: "Refresh token is required" });
  }

  try {
    const decoded = jwt.verify(refreshToken, process.env.JWT_REFRESH_SECRET);
    const user = await User.findById(decoded.id)
      .populate({
        path: "role",
        model: "Role",
        select: "name",
      })
      .populate({
        path: "assignedRoutes",
        model: "Route",
        select: "path name",
      })
      .populate({
        path: "menus",
        model: "Menu",
        populate: {
          path: "routes",
          model: "Route",
          select: "path name",
        },
      });

    if (!user || user.refreshToken !== refreshToken) {
      return res.status(403).json({ message: "Invalid refresh token" });
    }

    const { accessToken, refreshToken: newRefreshToken } = await generateTokens(
      user
    );

    user.refreshToken = newRefreshToken;
    await user.save();

    res.cookie("refreshToken", newRefreshToken, {
      httpOnly: true,
      secure: process.env.NODE_ENV === "production", // Use secure cookies in production
      sameSite: "strict", // Prevent CSRF attacks
      maxAge: 15 * 24 * 60 * 60 * 1000, // 15 days
    });
    res.status(200).json({ accessToken });
  } catch (err) {
    res.status(403).json({ message: "Invalid or expired refresh token", err });
  }
};
