const Cart = require("../models/cartModel");

// @desc    Add product to cart (with full frontend structure)
// @route   POST /api/cart

exports.addToCart = async (req, res) => {
  try {
    // Accepts array of product objects
    const products = Array.isArray(req.body) ? req.body : [req.body];
    // return console.log(req.user);
    // Validate input
    if (!products.length || !products.every((p) => p._id)) {
      return res.status(400).json({
        success: false,
        message: "Invalid product data format. Requires _id for all items",
      });
    }

    let cart;

    // For authenticated users
    if (req.user) {
      cart = await Cart.findOne({ user: req.user._id });
      if (!cart) {
        cart = new Cart({
          user: req.user._id,
          items: [],
        });
      }
    }
    // For guest users
    else {
      if (!req.session.cartId) {
        cart = new Cart({ items: [] });
        await cart.save();
        req.session.cartId = cart._id;
      } else {
        cart = await Cart.findById(req.session.cartId);
        if (!cart) {
          cart = new Cart({ items: [] });
          await cart.save();
          req.session.cartId = cart._id;
        }
      }
    }

    // Process each product - using for...of to maintain order
    for (const product of products) {
      // Convert both IDs to string for reliable comparison
      const productId = product._id.toString();

      // Find existing item
      const existingItem = cart.items.find(
        (item) => item._id.toString() === productId
      );

      if (existingItem) {
        // Safely increment count (handle undefined cases)
        existingItem.count = (existingItem.count || 1) + (product.count || 1);
        existingItem.updatedAt = new Date();
      } else {
        // Add new item with proper structure
        const newItem = {
          ...product,
          _id: product._id, // Ensure _id is properly set
          count: product.count || 1,
          addedAt: new Date(),
          updatedAt: new Date(),
        };
        cart.items.push(newItem);
      }
    }

    // Mark the items array as modified
    cart.markModified("items");
    await cart.save();

    // Calculate totals
    const totalItems = cart.items.reduce(
      (sum, item) => sum + (item.count || 1),
      0
    );
    const subtotal = cart.items.reduce(
      (sum, item) => sum + (item.count || 1) * (item.sellingPrice || 0),
      0
    );

    res.status(200).json({
      success: true,
      cartId: cart._id,
      totalItems,
      subtotal,
      items: cart.items.map((item) => ({
        _id: item._id,
        name: item.name,
        count: item.count,
        sellingPrice: item.sellingPrice,
        salesId: item.salesId,
        addedAt: item.addedAt,
        updatedAt: item.updatedAt,
      })),
    });
  } catch (error) {
    console.error("Cart Error:", error);
    res.status(500).json({
      success: false,
      message: "Server error while updating cart",
      error: error.message,
    });
  }
};

exports.removeFromCart = async (req, res) => {
  try {
    // Get product ID from route parameters
    const { productId } = req.params;

    // Validate input
    if (!productId) {
      return res.status(400).json({
        success: false,
        message: "Product ID is required as a route parameter",
      });
    }

    // Only allow authenticated users (removed guest cart functionality)
    if (!req.user) {
      return res.status(401).json({
        success: false,
        message: "Authentication required",
      });
    }

    // Find the user's cart
    const cart = await Cart.findOne({ user: req.user._id });
    if (!cart) {
      return res.status(404).json({
        success: false,
        message: "Cart not found",
      });
    }

    // Find item index
    const itemIndex = cart.items.findIndex(
      (item) => item._id.toString() === productId.toString()
    );

    // If item not found
    if (itemIndex === -1) {
      return res.status(404).json({
        success: false,
        message: "Item not found in cart",
      });
    }

    // Get item before removal for total calculations
    const removedItem = cart.items[itemIndex];

    // Remove the item
    cart.items.splice(itemIndex, 1);
    cart.updatedAt = new Date();
    await cart.save();

    // Calculate new totals
    const totalItems = cart.items.reduce((sum, item) => sum + item.count, 0);
    const subtotal = cart.items.reduce(
      (sum, item) => sum + item.count * (item.sellingPrice || 0),
      0
    );

    res.status(200).json({
      success: true,
      message: "Item removed successfully",
      cartId: cart._id,
      totalItems,
      subtotal,
      removedItem: {
        _id: removedItem._id,
        name: removedItem.name,
        count: removedItem.count,
        sellingPrice: removedItem.sellingPrice,
      },
      remainingItems: cart.items.map((item) => ({
        _id: item._id,
        name: item.name,
        count: item.count,
        sellingPrice: item.sellingPrice,
      })),
    });
  } catch (error) {
    console.error("Remove from cart error:", error);
    res.status(500).json({
      success: false,
      message: "Server error while removing item from cart",
      error: error.message,
    });
  }
};

exports.getCartItems = async (req, res) => {
  try {
    let cart;

    // For authenticated users
    if (req.user) {
      cart = await Cart.findOne({
        user: req.user._id,
        status: "active",
      }).populate("user", "name email"); // Optional: populate user details
    }
    // For guest users
    else {
      return res.status(200).json({
        success: true,
        message: "No active cart found",
        items: [],
        totalItems: 0,
        subtotal: 0,
      });
    }

    // If no cart found
    if (!cart) {
      return res.status(200).json({
        success: true,
        message: "No active cart found",
        items: [],
        totalItems: 0,
        subtotal: 0,
      });
    }

    // Calculate totals
    const totalItems = cart.items.reduce(
      (sum, item) => sum + (item.count || 0),
      0
    );
    const subtotal = cart.items.reduce(
      (sum, item) => sum + (item.count || 0) * (item.sellingPrice || 0),
      0
    );

    // Format response items
    const formattedItems = cart.items.map((item) => ({
      _id: item._id,
      name: item.name,
      count: item.count,
      sellingPrice: item.sellingPrice,
      costPrice: item.costPrice,
      quantity: item.quantity,
      expireDate: item.expireDate,
      category: item.category,
      salesId: item.salesId,
      addedAt: item.addedAt,
      updatedAt: item.updatedAt,
    }));

    res.status(200).json({
      success: true,
      cartId: cart._id,
      userId: cart.user,
      totalItems,
      subtotal,
      items: formattedItems,
      createdAt: cart.createdAt,
      updatedAt: cart.updatedAt,
    });
  } catch (error) {
    console.error("Get Cart Error:", error);
    res.status(500).json({
      success: false,
      message: "Server error while fetching cart items",
      error: error.message,
    });
  }
};
