const Category = require("../models/categoryModel");
const slugify = require("slugify");
const Activity = require("../models/activityTrackerModel");
const catchAsync = require("../utils/catchAsync");
const AppError = require("../utils/appError");
const { generate10DigitUUID } = require("../helpers/generater");

exports.createCategory = catchAsync(async (req, res, next) => {
  const { name } = req.body;
  let slug = slugify(name) + generate10DigitUUID();
  const alreadyExist = await Category.findOne({ slug });
  if (alreadyExist) {
    return next(new AppError("Category name already exist", 400));
  }

  let category = await new Category({
    name: name,
    slug,
  }).save();
  // const activity = new Activity({
  //   user: req.user._id,
  //   activityType: "category_added",
  //   details: {
  //     name: category.name,
  //   },
  // });
  // await activity.save();
  res.status(200).send(category);
});

// get all category
exports.getAllCategories = catchAsync(async (req, res, next) => {
  const { limit, search } = req.query;
  const parsedLimit = parseInt(limit);
  let query = {};

  // If there is a search query, update the query object
  if (search) {
    query = {
      name: { $regex: new RegExp(search, "i") }, // Case-insensitive search
    };
  }
  const category = await Category.find({})
    .sort({ createdAt: -1 })
    .limit(parsedLimit);
  // if (!data) return res.status(400).send({ error: 'Categories not found' });
  res.status(200).send(category);
});

exports.deleteCategory = catchAsync(async (req, res, next) => {
  const category = await Category.findById(req.params.id);
  if (!category) {
    return next(new AppError("Category not found with this ID", 404));
  }

  deletedcategory = await Category.findByIdAndDelete({ _id: category._id });
  const activity = new Activity({
    user: req.user._id,
    activityType: "category_deleted",
    details: {
      name: deletedcategory.name,
    },
  });
  await activity.save();
  res.status(200).json({
    success: true,
  });
});

exports.updateCategory = catchAsync(async (req, res, next) => {
  const { name } = req.body;
  let slug = slugify(name) + generate10DigitUUID();

  const found = await Category.findById(req.params.id);

  if (!found) {
    return next(new AppError("Category not found with this ID", 404));
  }

  const category = await Category.findByIdAndUpdate(
    req.params.id,
    { name, slug },
    { new: true }
  );
  const activity = new Activity({
    user: req.user._id,
    activityType: "category_updated",
    details: {
      name: category.name,
    },
  });
  await activity.save();
  res.status(200).json({
    success: true,
    category,
  });
});
