const mongoose = require("mongoose");
const Sales = require("../models/salesModel");
const DebtorsHistory = require("../models/debtorsHistoryModel");
const Product = require("../models/productModel"); // Assuming you have a Product model

exports.getMaxConsumption = async (req, res) => {
  const { period } = req.query;

  // Ensure period is a number
  const months = parseInt(period, 10);
  if (isNaN(months)) {
    return res.status(400).json({ message: "Invalid period" });
  }

  const endDate = new Date();
  const startDate = new Date();
  startDate.setMonth(endDate.getMonth() - months);

  try {
    const results = await Sales.aggregate([
      { $unwind: "$products" }, // Unwind the products array
      { $match: { dateTime: { $gte: startDate, $lte: endDate } } }, // Filter by date range
      {
        $group: {
          _id: "$products._id", // Group by product ID
          productName: { $first: "$products.name" }, // Get product name from sales data
          totalQuantitySold: { $sum: "$products.count" }, // Sum the quantities sold
          maxConsumption: { $max: "$products.count" }, // Maximum quantity sold in a single transaction
        },
      },
      {
        $project: {
          _id: 0,
          productId: "$_id",
          productName: 1, // Project the product name
          totalQuantitySold: 1, // Pass the totalQuantitySold from sales
          maxConsumption: 1, // Maximum consumption (max units sold in one transaction)
        },
      },
      { $sort: { totalQuantitySold: -1 } }, // Sort by highest consumption
    ]);

    return res.status(200).json(results);
  } catch (err) {
    console.error(err);
    return res
      .status(500)
      .json({ message: "Server error", error: err.message });
  }
};

exports.calculateMaxStock = async (req, res) => {
  const { period } = req.query;
  const endDate = new Date();
  const startDate = new Date();
  startDate.setMonth(endDate.getMonth() - period); // Calculate start date based on the period

  // Step 1: Calculate max consumption for each product from sales data
  const salesResults = await Sales.aggregate([
    { $unwind: "$products" }, // Unwind the products array in each sale
    { $match: { dateTime: { $gte: startDate, $lte: endDate } } }, // Filter by date range
    {
      $group: {
        _id: "$products._id", // Group by product ID
        totalQuantitySold: { $sum: "$products.count" }, // Total quantity sold of the product
        maxConsumption: { $max: "$products.count" }, // Maximum quantity sold in a single transaction
      },
    },
  ]);

  // Step 2: Fetch the current stock levels from the Product model
  const productIds = salesResults.map((result) => result._id); // Get product IDs from sales results
  const productStocks = await Product.find(
    { _id: { $in: productIds } },
    { _id: 1, name: 1, quantity: 1 }
  );

  // Step 3: Combine the sales results with current stock levels and calculate max stock required
  const finalResults = salesResults.map((sale) => {
    const product = productStocks.find((p) => p._id.equals(sale._id)); // Match product by ID
    return {
      productName: product ? product.name : "Unknown Product",
      currentStock: product ? product.quantity : 0, // Current stock level from the product model
      maxConsumption: sale.maxConsumption, // Max consumption from sales data
      maxStockRequired: Math.max(
        sale.maxConsumption,
        product ? product.quantity : 0
      ), // Max of stock or consumption
    };
  });

  return res.status(200).json(finalResults);
};

exports.calculateReorderLevelsForDays = async (req, res) => {
  const { period = 7, leadTime = 5, safetyStock = 10 } = req.query; // Default period is 7 days if not provided
  const endDate = new Date();
  const startDate = new Date();

  // Subtract the period in days from the current date
  startDate.setDate(endDate.getDate() - period);

  // Set startDate to the beginning of the day (00:00:00) and endDate to the next day's 00:00:00
  startDate.setUTCHours(0, 0, 0, 0); // Reset time to start of the day for startDate
  endDate.setUTCHours(0, 0, 0, 0); // Reset time to start of the day for endDate
  const nextDay = new Date(endDate);
  nextDay.setDate(nextDay.getDate() + 1); // Move to next day to include the whole day of endDate

  // Log start and end date for debugging purposes
  // console.log(
  //   "Start Date:",
  //   startDate,
  //   "End Date:",
  //   endDate,
  //   "Next Day:",
  //   nextDay
  // );

  // Step 1: Calculate max consumption for each product from sales data within the selected period
  const salesResults = await Sales.aggregate([
    { $unwind: "$products" }, // Unwind the products array in each sale
    { $match: { dateTime: { $gte: startDate, $lt: nextDay } } }, // Filter by the selected date range (ignore time)
    {
      $group: {
        _id: "$products._id", // Group by product ID
        totalQuantitySold: { $sum: "$products.count" }, // Total quantity sold of the product within the period
        maxConsumption: { $max: "$products.count" }, // Maximum quantity sold in a single transaction within the period
      },
    },
    { $sort: { _id: 1 } }, // Sort by product ID to ensure consistent ordering
  ]);

  // Step 2: Fetch the current stock levels from the Product model
  const productIds = salesResults.map((result) => result._id); // Get product IDs from sales results
  const productStocks = await Product.find(
    { _id: { $in: productIds } },
    { _id: 1, name: 1, quantity: 1 }
  ).sort({ _id: 1 }); // Sort by product ID to match sales results order

  // Step 3: Combine the sales results with current stock levels and calculate reorder level and restock suggestion
  const finalResults = salesResults.map((sale) => {
    const product = productStocks.find((p) => p._id.equals(sale._id)); // Match product by ID
    const consumption = sale.totalQuantitySold || 0;
    const maxConsumption = sale.maxConsumption || 0;
    const currentStock = product ? product.quantity : 0;

    // Calculate the reorder level
    const reorderLevel = maxConsumption * leadTime + safetyStock;

    // Calculate the suggested restock quantity (difference between reorder level and current stock)
    const restockQuantity =
      currentStock < reorderLevel ? reorderLevel - currentStock : 0;

    // Determine recommendation based on current stock
    const recommendation =
      restockQuantity > 0
        ? `Reorder needed: Suggest restocking ${restockQuantity} units.`
        : "Stock is sufficient.";

    return {
      productName: product ? product.name : "Unknown Product",
      currentStock, // Current stock level from the product model
      consumption, // Total consumption from sales data
      maxConsumption, // Max consumption from sales data
      reorderLevel, // Reorder level calculated based on consumption, lead time, and safety stock
      restockQuantity, // Suggested restock quantity
      recommendation, // Recommendation based on stock levels
    };
  });

  return res.status(200).json(finalResults);
};

exports.calculateReorderLevels = async (req, res) => {
  const { period = 1, leadTime = 5, safetyStock = 10 } = req.query; // Default period is 1 month
  const endDate = new Date();
  const startDate = new Date();
  startDate.setMonth(endDate.getMonth() - period); // Adjust start date based on the period

  try {
    // Step 1: Calculate max consumption for each product from both Sales and DebtorsSales data within the selected period
    const salesResults = await Sales.aggregate([
      { $unwind: "$products" }, // Unwind the products array in each sale
      { $match: { dateTime: { $gte: startDate, $lte: endDate } } }, // Filter by the selected date range
      {
        $group: {
          _id: "$products._id", // Group by product ID
          totalQuantitySold: { $sum: "$products.count" }, // Total quantity sold of the product
          maxConsumption: { $max: "$products.count" }, // Maximum quantity sold in a single transaction
        },
      },
      { $sort: { _id: 1 } }, // Sort by product ID
    ]);

    const debtorsResults = await DebtorsHistory.aggregate([
      { $unwind: "$products" }, // Unwind the products array in each debtor's sale
      { $match: { dateTime: { $gte: startDate, $lte: endDate } } }, // Filter by the selected date range
      {
        $group: {
          _id: "$products._id", // Group by product ID
          totalQuantitySold: { $sum: "$products.count" }, // Total quantity sold of the product
          maxConsumption: { $max: "$products.count" }, // Maximum quantity sold in a single transaction
        },
      },
      { $sort: { _id: 1 } }, // Sort by product ID
    ]);

    // Combine sales results from both normal sales and debtors' sales
    const combinedResults = [...salesResults, ...debtorsResults];

    // Merge the sales and debtors' sales data for the same products
    const mergedResults = combinedResults.reduce((acc, curr) => {
      const existingProduct = acc.find((item) =>
        new mongoose.Types.ObjectId(item._id).equals(
          new mongoose.Types.ObjectId(curr._id)
        )
      );

      if (existingProduct) {
        existingProduct.totalQuantitySold += curr.totalQuantitySold;
        existingProduct.maxConsumption = Math.max(
          existingProduct.maxConsumption,
          curr.maxConsumption
        );
      } else {
        acc.push(curr);
      }
      return acc;
    }, []);

    // Step 2: Fetch the current stock levels from the Product model
    const productIds = mergedResults.map((result) => result._id); // Get product IDs from the combined results
    const productStocks = await Product.find(
      { _id: { $in: productIds } },
      { _id: 1, name: 1, quantity: 1 }
    ).sort({ _id: 1 }); // Sort by product ID to match results order

    // Step 3: Combine the sales results with current stock levels and calculate reorder level and restock suggestion
    const finalResults = mergedResults
      .map((sale) => {
        const product = productStocks.find((p) =>
          new mongoose.Types.ObjectId(p._id).equals(
            new mongoose.Types.ObjectId(sale._id)
          )
        ); // Match product by ID
        if (!product || product.name === "Unknown Product") return null; // Skip products with unknown name or ID

        const consumption = sale.totalQuantitySold || 0;
        const maxConsumption = sale.maxConsumption || 0;
        const currentStock = product ? product.quantity : 0;

        // Calculate the reorder level
        const reorderLevel = maxConsumption * leadTime + safetyStock;

        // Calculate the suggested restock quantity (difference between reorder level and current stock)
        const restockQuantity =
          currentStock < reorderLevel ? reorderLevel - currentStock : 0;

        // Determine recommendation based on current stock
        const recommendation =
          restockQuantity > 0
            ? `Reorder needed: Suggest restocking ${restockQuantity} units.`
            : "Stock is sufficient.";

        return {
          productName: product.name,
          id: product._id,
          currentStock, // Current stock level from the product model
          consumption, // Total consumption from sales data
          maxConsumption, // Max consumption from sales data
          reorderLevel, // Reorder level calculated based on consumption, lead time, and safety stock
          restockQuantity, // Suggested restock quantity
          recommendation, // Recommendation based on stock levels
        };
      })
      .filter((result) => result !== null); // Filter out the null values (Unknown Products)

    return res.status(200).json(finalResults);
  } catch (error) {
    console.error(error);
    return res
      .status(500)
      .json({ message: "Server error", error: error.message });
  }
};
