const fs = require("fs");
const path = require("path");
const csv = require("fast-csv");
const Product = require("../models/productModel");
const Sales = require("../models/salesModel");
const TallyCard = require("../models/tallyCardModel");
const slugify = require("slugify");
const { generate10DigitUUID } = require("../helpers/generater");

exports.ImportProducts = async (req, res) => {
  const totalRecords = [];

  try {
    if (!req.file) {
      return res.status(400).json({ message: "No file uploaded" });
    }

    const filePath = path.join(
      __dirname,
      "../",
      "public/csv",
      req.file.filename
    );

    fs.createReadStream(filePath)
      .pipe(csv.parse({ headers: true }))
      .on("error", (error) => {
        console.error("Error reading CSV file:", error);
        return res.status(500).json({ message: "Error reading CSV file" });
      })
      .on("data", (row) => totalRecords.push(row))
      .on("end", async () => {
        try {
          const updatedOrCreatedProducts = [];

          for (const record of totalRecords) {
            // Validate required fields
            if (
              !record.name ||
              !record.quantity ||
              !record.costPrice ||
              !record.sellingPrice
            ) {
              // console.error("Missing required fields in record:", record);
              continue; // Skip this record and move to the next one
            }

            // Validate numeric fields
            const quantity = parseInt(record.quantity, 10);
            const costPrice = parseFloat(record.costPrice); // Use parseFloat for decimals
            const sellingPrice = parseFloat(record.sellingPrice); // Use parseFloat for decimals

            if (isNaN(quantity) || quantity < 0) {
              console.error("Invalid quantity in record:", record);
              continue;
            }
            if (isNaN(costPrice) || costPrice < 0) {
              console.error("Invalid costPrice in record:", record);
              continue;
            }
            if (isNaN(sellingPrice) || sellingPrice < 0) {
              console.error("Invalid sellingPrice in record:", record);
              continue;
            }

            const productData = {
              name: record.name,
              slug: slugify(record.name, { lower: true }),
              quantity: quantity,
              initialQty: quantity,
              costPrice: costPrice,
              sellingPrice: sellingPrice,
              user: req.user._id,
              hideExpireDate: false, // Default to false
            };

            // Add expireDate only if it's provided and valid
            if (record.expireDate && record.expireDate.trim() !== "") {
              const expireDate = new Date(record.expireDate);
              if (expireDate.toString() !== "Invalid Date") {
                productData.expireDate = expireDate;
                productData.hideExpireDate = true; // Set to true if expireDate is valid
              }
            }

            // Check if the product already exists
            const existingProduct = await Product.findOne({
              name: record.name,
            });

            if (existingProduct) {
              // Update existing product fields only if provided
              existingProduct.quantity = productData.quantity;
              existingProduct.costPrice = productData.costPrice;
              existingProduct.sellingPrice = productData.sellingPrice;
              existingProduct.hideExpireDate = productData.hideExpireDate;

              // Only update expireDate if it's provided
              if (productData.expireDate) {
                existingProduct.expireDate = productData.expireDate;
              }

              await existingProduct.save();
              updatedOrCreatedProducts.push(existingProduct);
            } else {
              // Create a new product
              const newProduct = await Product.create(productData);
              updatedOrCreatedProducts.push(newProduct);

              // Create a tally card entry for the new product
              await createTallyCardEntry(
                newProduct._id,
                newProduct.quantity,
                req.user._id
              );
            }
          }

          return res.status(200).json({
            message: "Products imported successfully",
            data: updatedOrCreatedProducts,
          });
        } catch (err) {
          console.error("Error importing products:", err);
          return res.status(500).json({
            message: "Error importing products",
            error: err.message,
          });
        } finally {
          // Clean up the uploaded file after processing
          fs.unlink(filePath, (err) => {
            if (err) console.error("Error deleting CSV file:", err);
          });
        }
      });
  } catch (error) {
    console.error("Error processing file upload:", error);
    return res.status(500).json({
      message: "Error processing file upload",
      error: error.message,
    });
  }
};

const createTallyCardEntry = async (productId, productQuantity, userId) => {
  try {
    const newTallyEntry = new TallyCard({
      product: productId,
      inflow: productQuantity,
      outflow: 0,
      stockBalance: productQuantity,
      previousQty: productQuantity,
      description: "Initial Stoke Take",
      user: userId,
      date: new Date(),
    });
    await newTallyEntry.save();
  } catch (error) {
    console.error("Error creating tally card entry:", error);
    throw error;
  }
};

exports.importSalesBackLog = async (req, res) => {
  try {
    // Define an array to store async functions for each row processing
    const promises = [];

    // Define customer and dateTime variables outside of the event handlers
    let customer = {};
    let dateTime;
    let discountSold = 0;
    let discount = [];
    // Function to handle each row of data
    // Function to handle each row of data
    const handleRow = async (row) => {
      try {
        // Check if productId is provided in the row
        if (!row.productId) {
          throw new Error(`productId is missing or invalid`);
        }

        // Find the corresponding product in the Product schema by ID
        const product = await Product.findById(row.productId);
        if (!product) {
          throw new Error(`Product not found for ID: ${row.productId}`);
        }

        // Retrieve the quantity of the product
        const productQuantity = product.quantity;

        // Calculate the quantity sold for the product
        const quantitySold = parseInt(row.quantitySold);

        const discount = parseFloat(row.discount) || 0;
        discountSold += discount; // Up

        // Construct an object representing the product with count
        const productWithCount = {
          ...product.toObject(),
          count: quantitySold,
          id: row.productId,
          discount: discount,
        };

        // Subtract quantitySold from product quantity and update the product in the database
        product.quantity -= quantitySold;
        await product.save();

        return { productWithCount, productQuantity, quantitySold }; // Return the processed product
      } catch (error) {
        console.error("Error:", error);
        throw error; // Propagate the error to the caller
      }
    };

    // Read the CSV file and parse its content
    fs.createReadStream(
      path.join(__dirname, "../", "/public/csv/", req.file.filename)
    )
      .pipe(csv.parse({ headers: true }))
      .on("error", (error) => {
        console.error("Error:", error);
        res.status(500).json({ message: "Internal server error" });
      })
      .on("data", async (row) => {
        // Extract customer information from the row
        customer = {
          name: row.customerName || "Unknown",
          phoneNumber: row.customerPhoneNumber || "Unknown",
        };

        // Extract dateTime information from the row
        dateTime = row.dateTime ? new Date(row.dateTime) : new Date();

        // Push the async function for handling the row into the promises array
        promises.push(() => handleRow(row));
      })
      .on("end", async () => {
        try {
          // Execute all async functions to update products in the database
          const salesData = await Promise.all(promises.map((p) => p()));

          await Promise.all(
            salesData.map(
              async ({ productWithCount, productQuantity, quantitySold }) => {
                try {
                  await createTallyCardEntryImportSales(
                    productWithCount.id,
                    productQuantity,
                    quantitySold,
                    productQuantity - quantitySold,
                    req.user._id
                  );
                } catch (err) {
                  console.error("Error creating tally card entry:", err);
                  throw err;
                }
              }
            )
          );

          // Calculate subTotal and grandTotal dynamically
          const subTotal = salesData.reduce(
            (acc, { productWithCount, quantitySold }) =>
              acc + productWithCount.sellingPrice * quantitySold,
            0
          );
          // Calculate the total discounted amount for each item
          const totalDiscountedAmounts = salesData.map((item) => {
            return (
              item.productWithCount.count * item.productWithCount.sellingPrice -
              item.productWithCount.discount
            );
          });

          const newGrandTotal = totalDiscountedAmounts.reduce(
            (total, amount) => {
              // Check if the current amount is a valid number
              if (typeof amount === "number" && !isNaN(amount)) {
                return total + amount;
              } else {
                // If the current amount is not a valid number, skip it
                return total; // Return the current total without adding the invalid amount
              }
            },
            0
          );

          const discountedAmounts = salesData.map(({ productWithCount }) => ({
            product: productWithCount.id,
            discount: productWithCount.discount,
          }));

          const grandTotal = subTotal - discountSold;

          // Create a new Sales document with accumulated data
          const sale = new Sales({
            customer: customer,
            products: salesData.map(({ productWithCount }) => productWithCount),
            dateTime: dateTime,
            subTotal: subTotal, // Calculated subtotal
            grandTotal: newGrandTotal, // Calculated grand total
            currency: "GH",
            receivedAmount: 520, // Hardcoded received amount
            paymentMethod: "Cash",
            balance: "0.00", // Hardcoded balance
            quantitySold: salesData.reduce(
              (acc, { quantitySold }) => acc + quantitySold,
              0
            ), // Total quantity sold
            invoiceID: generate10DigitUUID(), // Generate invoice ID
            discount: discountedAmounts,
            seller: req.user._id,
          });

          // Save the Sale document to the database
          await sale.save();

          res.status(200).json({ message: "server" });
        } catch (error) {
          console.error("Error:", error);
          res.status(500).json({ message: "Internal server error" });
        }
      });
  } catch (error) {
    console.error("Error:", error);
    res.status(500).json({ message: "Internal server error" });
  }
};

const createTallyCardEntryImportSales = async (
  productId,
  productQuantity,
  quantitySold,
  preQuantity,
  userId
) => {
  try {
    // console.log("previousQty***", productQuantity);
    const newTallyEntry = new TallyCard({
      product: productId,
      inflow: preQuantity,
      outflow: quantitySold,
      stockBalance: preQuantity,
      previousQty: productQuantity,
      description: "Bulk Sales Import",
      user: userId,
      date: new Date(),
    });
    await newTallyEntry.save();
  } catch (error) {
    console.error("Error creating tally card entry:", error);
    throw error;
  }
};
