const Customers = require("../models/customerModel");
const catchAsync = require("../utils/catchAsync");
const Sales = require("../models/salesModel");

exports.createCustomer = async (req, res) => {
  try {
    const customer = await new Customers({
      ...req.body,
    }).save();
    res.status(200).json(customer);
  } catch (err) {
    console.log(err);
    return res.status(400).send("customer create failed. Try again.");
  }
};

exports.getAllCustomers = async (req, res) => {
  const { limit, search } = req.query;
  const parsedLimit = parseInt(limit);
  let query = {};

  if (search) {
    const isNumeric = /^\d+$/.test(search); // Check if the search term is only numbers

    if (isNumeric) {
      query = {
        phone_number: { $regex: search.replace(/\D/g, ""), $options: "i" }, // Search only in phone_number
      };
    } else {
      query = {
        name: { $regex: search, $options: "i" }, // Search only in name
      };
    }
  }

  try {
    const allCustomers = await Customers.find(query)
      .sort({ createdAt: -1 })
      .limit(parsedLimit);
    res.status(200).send(allCustomers);
  } catch (err) {
    console.error("Error finding customers:", err);
    res.status(500).send({ message: "Error retrieving customers" });
  }
};

// get all category

exports.read = async (req, res) => {
  try {
    const customer = await Customers.findById({ _id: req.params.id }).exec();
    res.json(customer);
  } catch (err) {
    console.log(err);
  }
};

exports.updateCustomer = async (req, res) => {
  try {
    // const data = await Customers.findById({ _id: req.params.id });
    const customer = await Customers.findOneAndUpdate(
      { _id: req.params.id },
      req.body,
      {
        new: true,
      }
    ).exec();
    res.json(customer);
  } catch (err) {
    console.log(err);
    return res.status(400).send(err.message);
  }
};

exports.adminGetCustomerSales = catchAsync(async (req, res, next) => {
  const { customerId } = req.params;
  const { seller } = req.query;

  if (!customerId) {
    return res.status(400).json({ message: "Customer ID is required." });
  }

  let query = {
    customer: customerId,
  };

  if (seller) {
    // Parse seller as an array and get the first element (assuming only one seller is intended)
    const sellerId = Array.isArray(seller) ? seller[0] : seller;
    query.seller = sellerId;
  }

  if (req.query.startdate && req.query.enddate) {
    const startDate = new Date(req.query.startdate);
    startDate.setHours(0, 0, 0, 0);

    const endDate = new Date(req.query.enddate);
    endDate.setHours(23, 59, 59, 999);

    query.dateTime = { $gte: startDate, $lte: endDate };
  } else {
    return res
      .status(400)
      .json({ message: "Both startdate and enddate are required." });
  }

  const sales = await Sales.find(query)
    .populate("seller", "_id name")
    .populate("products", "_id name")
    .populate("customer", "_id name phone_number")
    .populate({
      path: "discount.product",
      model: "Product",
      select: "_id name",
    })
    .sort({ dateTime: -1 });

  if (!sales.length) {
    return res.status(200).json([]);
  }

  const grandTotal = sales.reduce((total, sale) => total + sale.grandTotal, 0);
  const quantitySold = sales.reduce(
    (total, sale) => total + sale.quantitySold,
    0
  );
  // grandTotal: grandTotal,
  // quantitySold: quantitySold,
  res.status(200).json(sales);
});
