const Debtor = require("../models/debtorsModel");
const DebtorsHistory = require("../models/debtorsHistoryModel");
const Activity = require("../models/activityTrackerModel");
const TallyCard = require("../models/tallyCardModel");
const salesReturns = require("../models/salesReturnsModel");
const Product = require("../models/productModel");
const catchAsync = require("../utils/catchAsync");
const mongoose = require("mongoose");
const {
  createDebtorsHistoryNotification,
} = require("./notificationController");

exports.createDebtor = async (req, res) => {
  try {
    const debtor = await new Debtor({
      ...req.body,
    }).save();
    res.json(debtor);
  } catch (err) {
    console.log(err);
    return res.status(400).send("Debtor create failed. Try again.");
  }
};

exports.getAllDebtors = async (req, res) => {
  const { limit, search } = req.query;
  const parsedLimit = parseInt(limit);
  let query = {};

  if (search) {
    const isNumeric = /^\d+$/.test(search); // Check if the search term is only numbers

    if (isNumeric) {
      query = {
        phone_number: { $regex: search.replace(/\D/g, ""), $options: "i" }, // Search only in phone_number
      };
    } else {
      query = {
        name: { $regex: search, $options: "i" }, // Search only in name
      };
    }
  }
  const all = await Debtor.find(query)
    .sort({ createdAt: -1 })
    .limit(parsedLimit);
  res.status(200).send(all);
};
// get all category

exports.read = async (req, res) => {
  try {
    const debtor = await Debtor.findById({ _id: req.params.id }).exec();
    res.json(debtor);
  } catch (err) {
    console.log(err);
  }
};

exports.updateDebtor = async (req, res) => {
  try {
    // const data = await Debtor.findById({ _id: req.params.id });
    const debtor = await Debtor.findOneAndUpdate(
      { _id: req.params.id },
      req.body,
      {
        new: true,
      }
    ).exec();
    res.json(debtor);
  } catch (err) {
    console.log(err);
    return res.status(400).send(err.message);
  }
};

exports.getDebtorHistory = async (req, res) => {
  try {
    let query = { debtor: req.params.id };

    let debtor = await DebtorsHistory.find(query)
      .populate("debtor", "name address phone_number")
      .sort({ dateTime: -1 });

    res.status(200).json(debtor);
  } catch (err) {
    console.log(err);
    res.status(500).json({ error: "Internal Server Error" });
  }
};

exports.addPaidHistory = async (req, res) => {
  try {
    const { paid_amount, date_paid } = req.body;
    const histories = await DebtorsHistory.findById(req.params.id);

    // const receivedAmount = histories?.grandTotal - subamount;
    // console.log(subamount);
    // console.log(histories?.grandTotal);
    // return console.log(histories?.grandTotal - subamount);
    const paidhistory = await DebtorsHistory.findOneAndUpdate(
      { _id: req.params.id },
      {
        $push: {
          paymentHistory: {
            paid_amount,
            date_paid,
          },
        },
      },
      { new: true }
    );
    const subamount = paidhistory?.paymentHistory.reduce(
      (acc, item) => acc + (item?.paid_amount || 0),
      0
    );
    const receivedAmount = histories?.grandTotal - subamount;
    await DebtorsHistory.findOneAndUpdate(
      { _id: req.params.id },
      { amount_remaining: receivedAmount },
      {
        new: true,
      }
    ).exec();

    res.status(200).json(paidhistory);
  } catch (err) {
    console.log(err);
    return res.status(400).send("Add paid history failed");
  }
};

exports.updatePaidHistory = async (req, res) => {
  try {
    const { _id, paid_amount, date_paid } = req.body;
    const histories = await DebtorsHistory.findOne({
      "paymentHistory._id": _id,
    });
    const receivedAmount = Number(histories?.grandTotal) - Number(paid_amount);
    // return console.log(receivedAmount);
    // return console.log(histories);
    const updated = await DebtorsHistory.updateOne(
      { "paymentHistory._id": _id },
      {
        $set: {
          "paymentHistory.$.paid_amount": paid_amount,
          "paymentHistory.$.date_paid": date_paid,
        },
      },
      { new: true }
    ).exec();

    await DebtorsHistory.findOneAndUpdate(
      { "paymentHistory._id": _id },
      { amount_remaining: receivedAmount },
      {
        new: true,
      }
    ).exec();
    // console.log("updated", updated);
    res.json({ ok: true });
  } catch (err) {
    console.log(err);
    return res.status(400).send("Update debtor history failed");
  }
};

exports.creditSalesByDate = catchAsync(async (req, res, next) => {
  const { seller, debtor } = req.query;

  // Check if debtor is provided and ensure it contains only one ID
  let debtorId = null;
  if (debtor) {
    // Parse debtor as an array and get the first element (assuming only one debtor is intended)
    const debtors = Array.isArray(debtor) ? debtor : [debtor];
    debtorId = debtors[0];
  }

  if (req.query.startdate && req.query.enddate) {
    startDate = new Date(req.query.startdate);
    startDate.setHours(0, 0, 0, 0);

    endDate = new Date(req.query.enddate);
    endDate.setHours(23, 59, 59, 999);
  } else {
    // beginning of current day
    var startDate = new Date();
    startDate.setHours(0, 0, 0, 0);

    // end of current day
    var endDate = new Date();
    endDate.setHours(23, 59, 59, 999);
  }

  let query = {
    dateTime: { $gte: startDate.toJSON(), $lte: endDate.toJSON() },
  };
  if (debtorId) {
    query.debtor = debtorId;
  }

  const sales = await DebtorsHistory.find(query)
    .populate("debtor", "_id name")
    .populate("seller", "_id name")
    .populate("products", "_id name")
    .populate("customer", "_id name phone_number")
    .populate({
      path: "discount.product",
      model: "Product", // Assuming your product model is named 'Product'
      select: "_id name", // Select the fields you want from the product model
    });

  // Check if sales is empty and return an empty array if no data is found for the selected debtor
  if (!sales.length && debtorId) {
    return res.status(200).json([]);
  }

  // var grandTotal = sales.reduce(function (p, c) {
  //   return p + c.grandTotal;
  // }, 0.0);

  // var quantitySold = sales.reduce(function (p, c) {
  //   return p + c.quantitySold;
  // }, 0.0);

  res.status(200).send(sales);
});

exports.createDebtorHistory = catchAsync(async (req, res, next) => {
  const {
    customer,
    grandTotal,
    quantitySold,
    paymentMethod,
    paidAmount,
    cart,
    subTotal,
    discount,
    currency,
    dateTime,
    debtor,
    invoiceID,
  } = req.body;

  let balanceTotal = Number(paidAmount) - Number(grandTotal).toFixed(2);

  const sanitizedDiscount = Object.fromEntries(
    Object.entries(discount).map(([productId, discountValue]) => [
      productId,
      discountValue === null ? 0 : discountValue,
    ])
  );
  const validCustomer =
    customer && mongoose.Types.ObjectId.isValid(customer) ? customer : null;

  const newSales = new DebtorsHistory({
    debtor,
    customer: validCustomer,
    seller: req.user._id,
    quantitySold,
    subTotal,
    grandTotal,
    paymentMethod,
    products: cart,
    receivedAmount: paidAmount,
    balance: balanceTotal,
    discount: Object.entries(sanitizedDiscount).map(
      ([productId, discountValue]) => ({
        product: productId,
        discount: Number(discountValue),
      })
    ),
    invoiceID,
    currency,
    dateTime,
  });

  // Create promises for updating product quantities and creating tally card entries
  const promises = cart?.map(async (item) => {
    const product = await Product.findById(item._id);
    if (!product) {
      throw new Error(`Product with ID ${item._id} not found`);
    }

    await createDebtorsHistoryNotification(req.user._id, newSales._id);
    await Promise.all([
      updateProductQuantity(item._id, item.count),
      createTallyCardEntryCreateSales(
        item._id,
        item.count,
        product.quantity,
        req.user._id,
        dateTime,
        item.salesId
      ),
    ]);
  });

  // Wait for all promises to resolve

  await newSales.save();
  await Promise.all(promises);

  const activity = new Activity({
    user: req.user._id,
    activityType: "sales",
    details: {
      customer,
      grandTotal,
      quantitySold,
      paymentMethod,
      paidAmount,
      cart,
      subTotal,
      discount,
      invoiceID: newSales.invoiceID,
      currency,
      dateTime: req.body.dateTime ? req.body.dateTime : new Date(),
    },
  });
  await activity.save();

  res.status(200).json({
    msg: "success",
    newSales,
  });
});

// Function to update product quantity
const updateProductQuantity = async (productId, soldQuantity) => {
  try {
    await Product.findOneAndUpdate(
      { _id: productId },
      {
        $inc: { quantity: -Number(soldQuantity) }, // Decrease quantity by soldQuantity
      }
    );
  } catch (error) {
    console.error("Error updating product quantity:", error);
    throw error;
  }
};

// Function to create entry in the tally card
const createTallyCardEntryCreateSales = async (
  productId,
  soldQuantity,
  productQuantity,
  userId,
  dateTime,
  salesId
) => {
  try {
    const newTallyEntry = new TallyCard({
      product: productId,
      inflow: 0,
      outflow: Number(soldQuantity),
      previousQty: Number(productQuantity),
      stockBalance: Number(productQuantity) - Number(soldQuantity),
      description: "Sales on credit",
      user: userId,
      date: dateTime ? dateTime : new Date(),
      salesId,
    });
    await newTallyEntry.save();
  } catch (error) {
    console.error("Error creating tally card entry:", error);
    throw error;
  }
};

exports.salesDebtorReturns = catchAsync(async (req, res, next) => {
  let {
    receiptId,
    salesId,
    newQuantity,
    previousQuantity,
    salesReturnReasons,
    sellingPrice,
  } = req.body;

  const { productId } = req.params;
  const product = await Product.findOne({ _id: productId });
  if (!product) {
    return next(new Error("Product not found"));
  }

  const sales = await DebtorsHistory.findById(receiptId);
  if (!sales) {
    return next(new Error("Sales record not found"));
  }

  const sale = sales.products.find((product) => product.salesId === salesId);
  if (!sale) {
    return next(new Error("Sales entry not found for the provided salesId"));
  }

  // Ensure newQuantity is defined before proceeding
  newQuantity = Number(newQuantity) || 0;

  const newUpdate = await Product.findOneAndUpdate(
    { slug: product.slug },
    {
      quantity: Number(product.quantity) + Number(newQuantity),
      sellingPrice: Number(sellingPrice),
    },
    {
      new: true,
    }
  );

  createTallyCardEntryUpdateSales(
    product._id,
    newQuantity,
    newUpdate.quantity,
    product.quantity,
    req.user._id
  );

  const updateResult = await DebtorsHistory.updateOne(
    { "products.salesId": salesId },
    {
      $set: {
        "products.$.count":
          sellingPrice === sale.sellingPrice
            ? Number(sale.count) - Number(newQuantity)
            : Number(sale.count),
        "products.$.salesreturnQty": Number(newQuantity),
        "products.$.previousQuantity": Number(sale.count),
        "products.$.salesReturnReasons": salesReturnReasons,
        "products.$.sellingPrice": Number(sellingPrice),
      },
    },
    {
      new: true,
    }
  );

  // Fetch the updated document
  const updatedProduct = await DebtorsHistory.findOne({
    "products.salesId": salesId,
  });
  if (!updatedProduct) {
    return next(new Error("Updated sales entry not found"));
  }

  const newGrandTotal = updatedProduct.products.reduce(
    (total, product) =>
      total + Number(product.sellingPrice) * Number(product.count),
    0
  );

  let grad = Number(sales.grandTotal) - Number(product.sellingPrice);
  let newQuantitySold = Number(sale.count) - newQuantity;

  await DebtorsHistory.findByIdAndUpdate(
    receiptId,
    {
      quantitySold:
        sellingPrice === Number(sale.sellingPrice)
          ? Number(sales.quantitySold) - Number(newQuantity)
          : Number(sales.quantitySold),
      grandTotal:
        Number(sales.quantitySold) !== Number(sales.quantitySold)
          ? grad
          : Number(newGrandTotal),
    },
    {
      new: true,
    }
  );

  const newSalesReturn = new salesReturns({
    product: productId,
    user: req.user._id,
    salesReturnReasons,
    newQuantity: Number(sale.count) - newQuantity,
    salesId,
    previousQuantity: Number(sale.count),
    salesreturnQty: Number(newQuantity),
  });
  await newSalesReturn.save();

  const activity = new Activity({
    user: req.user._id,
    activityType: "update_sales",
    details: {
      receiptId,
      salesId,
      newQuantity,
      previousQuantity,
      salesreturnQty: Number(newQuantity),
      salesReturnReasons,
    },
  });
  await activity.save();

  res.status(200).json(updateResult);
});

const createTallyCardEntryUpdateSales = async (
  productId,
  newQuantity,
  quantity,
  prevQty,
  userId
) => {
  try {
    const newTallyEntry = new TallyCard({
      product: productId,
      inflow: Number(newQuantity),
      outflow: 0,
      previousQty: Number(prevQty),
      stockBalance: Number(quantity),
      description: "Sales Return",
      user: userId,
      date: new Date(),
    });
    await newTallyEntry.save();
  } catch (error) {
    console.error("Error creating tally card entry:", error);
    throw error;
  }
};
