const Excess = require("../models/excessModel");
const catchAsync = require("../utils/catchAsync");

// POST /api/excess-pos
exports.createExcess = catchAsync(async (req, res) => {
  const { amount, date, note } = req.body;

  const entry = await Excess.create({
    amount,
    date: date || new Date(),
    note,
    user: req.user._id, // assuming middleware sets req.user
  });

  res.status(201).json(entry);
});

// controllers/excess
exports.getAllExcess = async (req, res) => {
  try {
    const { year, month, period } = req.query;
    const filter = {};

    // Handle year filter
    if (year) {
      filter.date = {
        $gte: new Date(`${year}-01-01`),
        $lte: new Date(`${year}-12-31T23:59:59.999Z`),
      };
    }

    // Handle month filter (requires year)
    if (month && year) {
      const monthInt = parseInt(month);
      const startDate = new Date(year, monthInt - 1, 1);
      const endDate = new Date(year, monthInt, 0, 23, 59, 59, 999);

      filter.date = {
        $gte: startDate,
        $lte: endDate,
      };
    }

    // Handle period filter
    if (period) {
      const now = new Date();

      switch (period.toLowerCase()) {
        case "today":
          filter.date = {
            $gte: new Date(now.setHours(0, 0, 0, 0)),
            $lte: new Date(now.setHours(23, 59, 59, 999)),
          };
          break;

        case "yesterday":
          const yesterday = new Date();
          yesterday.setDate(yesterday.getDate() - 1);
          filter.date = {
            $gte: new Date(yesterday.setHours(0, 0, 0, 0)),
            $lte: new Date(yesterday.setHours(23, 59, 59, 999)),
          };
          break;

        case "thisweek":
          const startOfWeek = new Date();
          startOfWeek.setDate(startOfWeek.getDate() - startOfWeek.getDay());
          const endOfWeek = new Date(startOfWeek);
          endOfWeek.setDate(startOfWeek.getDate() + 6);

          filter.date = {
            $gte: new Date(startOfWeek.setHours(0, 0, 0, 0)),
            $lte: new Date(endOfWeek.setHours(23, 59, 59, 999)),
          };
          break;

        case "last30days":
          const thirtyDaysAgo = new Date();
          thirtyDaysAgo.setDate(thirtyDaysAgo.getDate() - 30);

          filter.date = {
            $gte: thirtyDaysAgo,
            $lte: new Date(),
          };
          break;
      }
    }

    // Get filtered records
    const excessRecords = await Excess.find(filter)
      .populate("user", "name email")
      .sort({ date: -1 });

    // Calculate accumulated total
    const totalAmount = excessRecords.reduce(
      (sum, record) => sum + record.amount,
      0
    );

    res.json({
      success: true,
      count: excessRecords.length,
      totalAmount,
      data: excessRecords,
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      error: error.message,
    });
  }
};
exports.updateExcess = async (req, res) => {
  try {
    const { amount, note } = req.body;

    const excess = await Excess.findById(req.params.id);
    if (!excess) {
      return res.status(404).json({ error: "Excess record not found" });
    }

    // Only allow certain fields to be updated
    if (amount !== undefined) excess.amount = amount;
    if (note !== undefined) excess.note = note;

    await excess.save();
    res.json(excess);
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};
