const Expenses = require("../models/expensesModel");
const Salary = require("../models/salaryModel");
const Activity = require("../models/activityTrackerModel");
const catchAsync = require("../utils/catchAsync");
const AppError = require("../utils/appError");
const slugify = require("slugify");
const { generate10DigitUUID } = require("../helpers/generater");

// create contact Expenses
exports.createExpenses = catchAsync(async (req, res) => {
  const { name, amount, expenseDate } = req.body;
  let slug = slugify(name) + generate10DigitUUID();

  const expenses = await Expenses({
    slug,
    name,
    amount,
    user: req.user._id,
    date: expenseDate,
  }).save();
  const activity = new Activity({
    user: req.user._id,
    activityType: "expenses_added",
    details: {
      name,
      amount,
      expenseDate,
    },
  });
  await activity.save();
  res.status(200).send(expenses);
});

// get all contact Expenses
exports.getAllExpenses = catchAsync(async (req, res) => {
  const expenses = await Expenses.find({}).sort({ createdAt: -1 });
  res.status(200).send(expenses);
});

// get single contact Expenses
exports.getSingleExpenses = catchAsync(async (req, res, next) => {
  const { slug } = req.params;
  const expenses = await Expenses.findOne({ slug }).populate(
    "user",
    "name contactNum"
  );
  if (!expenses) {
    return next(new AppError("Expenses not found", 404));
  }
  res.status(200).send(expenses);
});

// update expeses
exports.updateExpenses = catchAsync(async (req, res, next) => {
  const { name, amount, expenseDate } = req.body;
  const { slug } = req.params;
  const expesesData = await Expenses.findOne({ slug });
  const updatedExpenses = await Expenses.findOneAndUpdate(
    { slug: expesesData.slug },
    {
      slug:
        name === expesesData?.name
          ? expesesData?.name
          : slugify(name) + generate10DigitUUID(),
      name: name,
      amount: amount,
      date: expenseDate,
      ...req.body,
    },
    {
      new: true,
    }
  );
  if (expesesData.salary) {
    await Salary.findByIdAndUpdate(
      { _id: expesesData.salary },
      { amount: amount },
      {
        new: true,
        runValidators: true,
      }
    );
  }
  const activity = new Activity({
    user: req.user._id,
    activityType: "expenses_updated",
    details: {
      name,
      amount,
      date: expenseDate,
    },
  });
  const activitySalary = new Activity({
    user: req.user._id,
    activityType: "salary_updated",
    details: {
      name: "Salary",
      amount,
      date: expenseDate,
    },
  });
  await activity.save();
  await activitySalary.save();
  res.status(200).send(updatedExpenses);
});

// delete Expenses
exports.deleteExpenses = catchAsync(async (req, res, next) => {
  const expense = await Expenses.findById(req.params.id);
  if (!expense) {
    return next(new AppError("Expense not found", 404));
  }

  const deletedExpenses = await Expenses.findByIdAndDelete({
    _id: expense._id,
  });
  const activity = new Activity({
    user: req.user._id,
    activityType: "expenses_deleted",
    details: {
      name: deletedExpenses.name,
      amount: deletedExpenses.amount,
      expenseDate: deletedExpenses.date,
    },
  });
  await activity.save();
  res.status(200).send({ status: "Success" });
});

exports.totalExpensesForSelectedDays = catchAsync(async (req, res, next) => {
  // if date is provided
  const { expensesStartDate, expensesEndDate } = req.body;
  if (expensesStartDate && expensesEndDate) {
    startDate = new Date(expensesStartDate);
    startDate.setHours(0, 0, 0, 0);

    endDate = new Date(expensesEndDate);
    endDate.setHours(23, 59, 59, 999);
  } else {
    // beginning of current day
    var startDate = new Date();
    startDate.setHours(0, 0, 0, 0);

    // end of current day
    var endDate = new Date();
    endDate.setHours(23, 59, 59, 999);
  }

  const docs = await Expenses.find({
    date: { $gte: startDate.toJSON(), $lte: endDate.toJSON() },
  });

  var result = {
    date: startDate,
  };

  if (docs) {
    var amount = docs.reduce(function (p, c) {
      return p + c.amount;
    }, 0.0);
    result.amount = parseFloat(parseFloat(amount).toFixed(2));
    return res.status(200).json(amount);
  } else {
    result.amount = 0;
    return res.status(200).json(result);
  }
});
