const { generateTokens } = require("../utils/token.js");
const Role = require("../models/roleModel.js");
const Route = require("../models/routeModel.js");
const User = require("../models/userModel.js");
const Menu = require("../models/menuModel.js");
const bcrypt = require("bcrypt");

exports.loadUser = async (req, res, next) => {
  if (!req.user) {
    return res.status(401).json({ message: "Unauthorized" });
  }
  try {
    const user = await User.findById(req.user.id)
      .populate({
        path: "role",
        model: "Role",
        populate: {
          path: "routes",
          model: "Route",
          select: "path name",
        },
      })
      .populate({
        path: "assignedRoutes",
        model: "Route",
        select: "path name",
      });

    if (!user) {
      return res.status(404).json({ message: "User not found" });
    }

    req.user = user;
    next();
  } catch (err) {
    console.error("Error loading user:", err);
    res.status(500).json({ message: "Internal Server Error" });
  }
};

exports.createMenu = async (req, res) => {
  const { name } = req.body;
  try {
    const newMenu = new Menu({ name });
    await newMenu.save();
    res.status(201).json(newMenu);
  } catch (err) {
    res.status(500).json({ message: "Error creating menu", err });
  }
};

exports.createRole = async (req, res) => {
  const { name, routeIds } = req.body;
  try {
    const routes = await Route.find({ _id: { $in: routeIds } });

    const newRole = new Role({ name, routes });
    await newRole.save();

    res.status(201).json(newRole);
  } catch (err) {
    res.status(500).json({ message: "Error creating role", err });
  }
};

exports.getAllRoles = async (req, res) => {
  try {
    const roles = await Role.find({});
    res.status(201).json(roles);
  } catch (err) {
    res.status(500).json({ message: "Error creating role", err });
  }
};
// Get all routes
exports.getAllRoutes = async (req, res) => {
  try {
    const routes = await Route.find({
      // $or: [{ menu: { $exists: false } }, { menu: null }],
    }).sort({ name: 1 }); // 1 for ascending order, -1 for descending order

    res.status(200).json(routes);
  } catch (err) {
    res.status(500).json({ message: "Error fetching routes", err });
  }
};
exports.getAllMenus = async (req, res) => {
  try {
    const menus = await Menu.find({}).populate({
      path: "routes",
      model: "Route",
      select: "path name",
    });
    res.status(201).json(menus);
  } catch (err) {
    res.status(500).json({ message: "Error creating menu", err });
  }
};

exports.createUser = async (req, res) => {
  const { password, email, username, roleId, routeIds } = req.body;
  try {
    let user = null;

    user = await User.findOne({ email });
    if (user) {
      return res.status(400).json({ message: "User already exist." });
    }
    const role = await Role.findById(roleId).populate("routes");
    const routes = await Route.find({ _id: { $in: routeIds } });

    if (!role || !routes) {
      return res.status(404).json({ message: "Route, or Role not found" });
    }
    // hash password
    const salt = await bcrypt.genSalt(10);
    const hashPassword = await bcrypt.hash(password, salt);
    user = new User({
      email,
      username,
      role: roleId,
      assignedRoutes: routeIds,
      password: hashPassword,
    });

    await user.save();
    res
      .status(200)
      .json({ success: true, message: "User successfully created" });
  } catch (err) {
    console.log(err);
  }
};

exports.createRoute = async (req, res) => {
  const { path, name, menuId } = req.body;
  try {
    const user = await User.findById(req.user.id);
    if (!user) {
      return res.status(404).json({ message: "User not found" });
    }
    // Create new route
    const newRoute = new Route({ path, name, menu: menuId });
    await newRoute.save();

    // Update the menu with the new route if menuId is provided
    if (menuId) {
      const menu = await Menu.findById(menuId);
      if (menu) {
        menu.routes.push(newRoute._id);
        await menu.save();
      } else {
        return res.status(404).json({ message: "Menu not found" });
      }
    }
    // Find the admin role and update its routes
    const adminRole = await Role.findOne({ name: "admin" });
    if (adminRole) {
      adminRole.routes.push(newRoute._id);
      // user.assignedRoutes.push(newRoute._id);
      await adminRole.save();
      // await user.save();
    }

    res.status(201).json(newRoute);
  } catch (err) {
    console.error("Error creating route:", err);
    res.status(500).json({ message: "Error creating route", err });
  }
};

exports.assignRouteToMenu = async (req, res) => {
  const { routeId, menuId } = req.body;
  try {
    const menu = await Menu.findById(menuId);
    const route = await Route.findById(routeId);
    if (!menu) {
      return res.status(404).json({ message: "menu not found" });
    }
    if (!route) {
      return res.status(404).json({ message: "route not found" });
    }
    const routeExistsInRoute = menu.routes.some((r) => r._id.equals(route._id));
    if (!routeExistsInRoute) {
      menu.routes.push(routeId);
      await menu.save();
    }
    const newRoute = await Route.findByIdAndUpdate(
      routeId,
      { menu: menuId },
      {
        new: true,
      }
    );

    res.status(201).json({ message: "success", newRoute });
  } catch (err) {
    console.error("Error creating route:", err);
    res.status(500).json({ message: "Error creating route", err });
  }
};
exports.assignMenuToUser = async (req, res) => {
  const { userId, menuId } = req.body;
  try {
    const menu = await Menu.findById(menuId);
    const user = await User.findById(userId);
    if (!menu) {
      return res.status(404).json({ message: "menu not found" });
    }
    if (!user) {
      return res.status(404).json({ message: "user not found" });
    }

    const menuExistsInUser = user.menus.some((r) => r._id.equals(menu._id));
    // return console.log(menuExistsInUser);
    if (!menuExistsInUser) {
      user.menus.push(menuId);
      await user.save();
    }
    // const newRoute = await Route.findByIdAndUpdate(
    //   routeId,
    //   { menu: menuId },
    //   {
    //     new: true,
    //   }
    // );

    res.status(201).json({ message: "success" });
  } catch (err) {
    console.error("Error creating route:", err);
    res.status(500).json({ message: "Error creating route", err });
  }
};

exports.assignRolesRoutesMenus = async (req, res) => {
  const { userId, roleId, routeId, menuId } = req.body;
  try {
    const user = await User.findById(userId);
    const role = await Role.findById(roleId).populate("routes");

    if (!user || !role) {
      return res.status(404).json({ message: "User or role not found" });
    }
    // Validate and fetch routes
    const routes = await Route.find({ _id: { $in: routeId } });
    if (routes.length !== routeId.length) {
      return res.status(404).json({ message: "Some routes not found" });
    }
    // Validate and fetch menus
    const menus = await Menu.find({ _id: { $in: menuId } }).populate("routes");
    if (menus.length !== menuId.length) {
      return res.status(404).json({ message: "Some menus not found" });
    }
    // Add the routes to the role if they're not already present
    routes.forEach((route) => {
      const routeExistsInRole = role.routes.some((r) =>
        r._id.equals(route._id)
      );
      if (!routeExistsInRole) {
        role.routes.push(route._id);
      }
    });
    await role.save();
    // Assign the role to the user
    user.role = role._id;

    // Update user's assigned routes and menus
    user.assignedRoutes = routeId;
    user.menus = menuId;

    // Rebuild combinedRoutes based on current assignedRoutes and menus
    const newCombinedRoutes = [];

    // Function to add routes to combinedRoutes if they do not exist
    const addToCombinedRoutes = (item, type, parentMenu = null) => {
      if (!newCombinedRoutes.some((r) => r._id.equals(item._id))) {
        newCombinedRoutes.push({
          ...item.toObject(),
          type,
          parentMenu,
        });
      }
    };

    // Add routes from assignedRoutes first
    const assignedRoutes = await Route.find({ _id: { $in: routeId } });
    assignedRoutes.forEach((route) => addToCombinedRoutes(route, "route"));

    // Then add menus and their submenus to combinedRoutes
    menus.forEach((menu) => {
      // Add the menu itself
      addToCombinedRoutes(menu, "menu");
      // Add each route associated with the menu as a submenu
      menu.routes.forEach((route) =>
        addToCombinedRoutes(route, "submenu", menu._id)
      );
    });

    // Update combinedRoutes in the user document
    user.combinedRoutes = newCombinedRoutes;
    user.isActiveCombineRoute = true;
    await user.save();

    // Generate new tokens
    const { accessToken, refreshToken } = generateTokens(user);
    user.refreshToken = refreshToken;

    res.status(200).json({
      message: "Role assigned and routes/menus updated successfully",
      user: {
        _id: user._id,
        username: user.username,
        role: user.role,
        assignedRoutes: user.assignedRoutes,
        menus: user.menus,
        combinedRoutes: user.combinedRoutes,
        refreshToken: user.refreshToken,
      },
      accessToken,
    });
  } catch (err) {
    console.error(err.message);
    res.status(500).json({ message: err.message });
  }
};

exports.removeMenuFromRoute = async (req, res) => {
  const { routeId, menuId } = req.body;
  try {
    const menu = await Menu.findById(menuId);
    const route = await Route.findById(routeId);

    if (!menu || !route) {
      return res.status(404).json({ message: "Route, or Menu not found" });
    }
    const routeExistsInMenu = menu.routes.some((r) => r._id.equals(route._id));
    if (routeExistsInMenu) {
      menu.routes.pull(routeId);
      await route.save();
      await menu.save();
      route.menu = null;
    }

    res.status(200).json({
      message: "Success",
      route,
    });
  } catch (err) {
    console.error("Error removing menu:", err);
    res.status(500).json({ message: "Error removing menu", err });
  }
};

exports.removeRouteFromRole = async (req, res) => {
  const { roleId, routeId } = req.body;
  try {
    const role = await Role.findById(roleId).populate("routes");
    const route = await Route.findById(routeId);
    if (!role || !route) {
      return res.status(404).json({ message: "Route, or role not found" });
    }
    // find the route to the role if it's not already present
    const routeExistsInRole = role.routes.some((r) => r._id.equals(route._id));
    // console.log(routeExistsInRole);
    if (routeExistsInRole) {
      role.routes.pull(route._id);
      await role.save();
    }

    res.status(200).json({
      message: "Success",
    });
  } catch (err) {
    console.error("Error assigning role:", err);
    res.status(500).json({ message: "Error assigning role", err });
  }
};
exports.removeRouteFromUser = async (req, res) => {
  const { userId, routeId } = req.body;
  try {
    const user = await User.findById(userId);
    const route = await Route.findById(routeId);

    if (!user || !route) {
      return res.status(404).json({ message: "User, route not found" });
    }

    //fined Assign the role to the user and add the new route to user's assigned routes if not already present

    if (user.assignedRoutes.some((r) => r.equals(route._id))) {
      user.assignedRoutes.pull(route._id);
    }
    await user.save();

    // Populate the user data
    await user.populate({
      path: "role",
      populate: {
        path: "routes",
        model: "Route",
        select: "path name",
      },
    });

    await user.populate({
      path: "assignedRoutes",
      model: "Route",
      select: "path name",
    });
    // Generate new tokens
    const { accessToken, refreshToken } = generateTokens(user);
    user.refreshToken = refreshToken;
    await user.save();

    res.status(200).json({
      message: "Role assigned and route added successfully",
      user: {
        _id: user._id,
        username: user.username,
        role: user.role,
        assignedRoutes: user.assignedRoutes,
        refreshToken: user.refreshToken,
      },
      accessToken,
    });
  } catch (err) {
    console.error("Error assigning role:", err);
    res.status(500).json({ message: "Error assigning role", err });
  }
};

// Get a specific route by ID
exports.getRouteById = async (req, res) => {
  const { id } = req.params;
  try {
    const route = await Route.findById(id);
    if (!route) {
      return res.status(404).json({ message: "Route not found" });
    }
    res.status(200).json(route);
  } catch (err) {
    res.status(500).json({ message: "Error fetching route", err });
  }
};

// Update a route by ID
exports.updateRouteById = async (req, res) => {
  const { id } = req.params;
  const { path, name } = req.body;
  try {
    const route = await Route.findByIdAndUpdate(
      id,
      { path, name },
      { new: true }
    );
    if (!route) {
      return res.status(404).json({ message: "Route not found" });
    }
    res.status(200).json(route);
  } catch (err) {
    res.status(500).json({ message: "Error updating route", err });
  }
};

// Delete a route by ID
exports.deleteRouteById = async (req, res) => {
  const { id } = req.params;
  try {
    const route = await Route.findByIdAndDelete(id);
    if (!route) {
      return res.status(404).json({ message: "Route not found" });
    }
    res.status(200).json({ message: "Route deleted successfully" });
  } catch (err) {
    res.status(500).json({ message: "Error deleting route", err });
  }
};

exports.updateUserCombinedRoutes = async (req, res) => {
  const { combinedRoutes } = req.body;
  const userId = req.user.id; // Make sure you have the user ID available

  try {
    // Retrieve the current user data
    const currentUser = await User.findById(userId);
    if (!currentUser) {
      return res.status(404).json({ message: "User not found" });
    }

    // Update the user with the new combinedRoutes data
    const updatedUser = await User.findByIdAndUpdate(
      userId,
      { $set: { combinedRoutes } },
      { new: true, runValidators: true }
    );

    if (!updatedUser) {
      return res.status(404).json({ message: "User not found" });
    }

    res.status(200).json({
      message: "User menu items updated successfully",
      data: updatedUser.combinedRoutes,
    });
  } catch (error) {
    console.error("Error updating user menu items:", error); // Log the error details
    res.status(500).json({ message: "Server error", error: error.message });
  }
};

exports.getUserCombinedRoutes = async (req, res) => {
  const userId = req.user.id; // Make sure you have the user ID available
  try {
    // Retrieve the current user data
    const currentUser = await User.findById(userId);
    if (!currentUser) {
      return res.status(404).json({ message: "User not found" });
    }

    // Reorder combinedRoutes to ensure routes come first
    const combinedRoutes = currentUser.combinedRoutes.sort((a, b) => {
      if (a.type === "route" && b.type !== "route") return -1;
      if (a.type !== "route" && b.type === "route") return 1;
      return 0;
    });
    // console.log("from getUserCombinedRoutes", combinedRoutes);

    res.status(200).json(combinedRoutes);
  } catch (error) {
    console.error("Error retrieving user combined routes:", error); // Log the error details
    res.status(500).json({ message: "Server error", error: error.message });
  }
};
