const Sales = require("../models/salesModel");
const DebtorsHistory = require("../models/debtorsHistoryModel");
const UserPreferences = require("../models/userPreferencesModel");
const PushSubscription = require("../models/pushSubscriptionModel");
const Notification = require("../models/notificationModel");
const User = require("../models/userModel");
const { sendPushNotification } = require("../services/pushService.js");
const { sendEmailNotification } = require("../services/emailService.js");
const { sendSingleNotification } = require("../socket/socketHandlers.js");
const {
  newSaleNotification,
  dailySalesSummary,
  productExpiryAlert,
  outOfStockAlert,
  productSoonExpiryAlert,
  salaryPaymentReminderTemplate,
} = require("../utils/emailTemplates");
// Create sale notification

exports.createSaleNotification = async (userId, saleId) => {
  try {
    const preferences = await UserPreferences.findOne({ user: userId });
    if (!preferences) {
      console.log(`No preferences found for user ${userId}`);
      return null;
    }

    const sale = await Sales.findById({ _id: saleId });
    if (!sale) {
      throw new Error(`Sale ${saleId} not found`);
    }

    const message = `New sale recorded. Invoice ID: ${
      sale.invoiceID
    }, Amount: GHS ${sale.grandTotal.toFixed(2)}`;
    // Create notification
    const notification = new Notification({
      user: userId,
      title: "New Sale",
      type: "sale",
      message,
      saleId: sale._id,
    });

    // Save it before emitting
    await notification.save();

    // ✅ Emit to user via Socket
    if (typeof sendSingleNotification === "function") {
      sendSingleNotification(userId, notification);
    } else {
      console.warn("sendSingleNotification is not defined or not a function.");
    }

    // ✅ Push Notification
    if (preferences?.notificationPreferences?.pushEnabled) {
      await sendPushNotification(userId, {
        title: "📊 New Sale",
        body: message,
      });
    }

    // ✅ Email Notification
    if (preferences.notificationPreferences?.emailEnabled) {
      await sendEmailNotification(userId, newSaleNotification(sale));
    }

    return notification;
  } catch (error) {
    console.error("Error creating sale notification:", error);
    throw error; // Still rethrow to handle elsewhere
  }
};
exports.createDebtorsHistoryNotification = async (userId, saleId) => {
  try {
    const preferences = await UserPreferences.findOne({ user: userId });
    if (!preferences) {
      console.log(`No preferences found for user ${userId}`);
      return null;
    }

    const sale = await DebtorsHistory.findById({ _id: saleId });
    if (!sale) {
      throw new Error(`Sale ${saleId} not found`);
    }

    const message = `New sale recorded. Invoice ID: ${
      sale.invoiceID
    }, Amount: GHS ${sale.grandTotal.toFixed(2)}`;
    // Create notification
    const notification = new Notification({
      user: userId,
      title: "New Sale",
      type: "sale",
      message,
      saleId: sale._id,
    });

    // Save it before emitting
    await notification.save();

    // ✅ Emit to user via Socket
    if (typeof sendSingleNotification === "function") {
      sendSingleNotification(userId, notification);
    } else {
      console.warn("sendSingleNotification is not defined or not a function.");
    }

    // ✅ Push Notification
    if (preferences?.notificationPreferences?.pushEnabled) {
      await sendPushNotification(userId, {
        title: "📊 New Sale",
        body: message,
      });
    }

    // ✅ Email Notification
    if (preferences.notificationPreferences?.emailEnabled) {
      await sendEmailNotification(userId, newSaleNotification(sale));
    }

    return notification;
  } catch (error) {
    console.error("Error creating sale notification:", error);
    throw error; // Still rethrow to handle elsewhere
  }
};
// Send daily sales summary
exports.sendDailySalesSummary = async (userId) => {
  try {
    const user = await User.findById(userId);
    if (!user) {
      console.warn(`User ${userId} not found`);
      return;
    }
    const preferences = await UserPreferences.findOne({ user: userId });
    if (!preferences || !preferences.notificationPreferences?.dailySummary) {
      console.warn(
        `No preferences or daily summary disabled for user ${userId}`
      );
      return;
    }

    const startOfDay = new Date();
    startOfDay.setHours(0, 0, 0, 0);
    const endOfDay = new Date();
    endOfDay.setHours(23, 59, 59, 999);
    const sales = await Sales.find({
      seller: userId,
      createdAt: { $gte: startOfDay, $lte: endOfDay },
    });
    const totalSales = sales.length;
    const totalAmount = sales.reduce((sum, sale) => sum + sale.grandTotal, 0);

    const message = `Daily Sales Summary: ${totalSales} sale(s) completed, totaling GHS ${totalAmount.toFixed(
      2
    )}`;

    // Save to notifications log
    const notification = await Notification.create({
      user: userId,
      title: "Daily Sales Summary",
      type: "daily_summary",
      message,
    });

    if (typeof sendSingleNotification === "function") {
      sendSingleNotification(userId, notification);
    } else {
      console.warn("sendSingleNotification is not defined or not a function.");
    }
    // Push notification
    if (preferences.notificationPreferences.pushEnabled) {
      await sendPushNotification(userId, {
        title: "📊 Daily Sales Summary",
        body: message,
      });
    }
    // Email notification
    if (preferences.notificationPreferences.emailEnabled && user.email) {
      await sendEmailNotification(userId, dailySalesSummary(sales));
    }
  } catch (error) {
    console.error(`❌ Error sending summary for user ${userId}:`, error);
  }
};

// Get user notifications
exports.getUserNotifications = async (req, res) => {
  try {
    const notifications = await Notification.find({ user: req.user._id })
      .sort({ createdAt: -1 })
      .limit(50);

    res.json(notifications);
  } catch (error) {
    res.status(500).json({ message: "Error fetching notifications" });
  }
};

// Update notification preferences
exports.updateNotificationPreferences = async (req, res) => {
  try {
    const { updates } = req.body;
    const preferences = await UserPreferences.findOneAndUpdate(
      { user: req.user._id },
      {
        notificationPreferences: updates.notificationPreferences,
        dailySummaryTime: updates.dailySummaryTime,
        providedEmail: updates.providedEmail,
        salaryReminderDate: updates.salaryReminderDate,
        salaryReminderTime: updates.salaryReminderTime,
      },
      { new: true, upsert: true }
    );
    // console.log(preferences);
    res.status(200).json(preferences);
  } catch (error) {
    res.status(500).json({ message: "Error updating preferences" });
  }
};

exports.registerPushSubscription = async (req, res) => {
  try {
    const { subscription } = req.body;

    // Validate subscription
    if (!subscription || !subscription.endpoint || !subscription.keys) {
      return res.status(400).json({ message: "Invalid subscription object" });
    }

    // Check if subscription already exists for this endpoint
    const existingSubscription = await PushSubscription.findOne({
      endpoint: subscription.endpoint,
      user: req.user._id,
    });

    if (existingSubscription) {
      return res.status(200).json({
        message: "Subscription already exists",
        subscription: existingSubscription,
      });
    }

    // Create new PushSubscription document
    const newSubscription = await PushSubscription.create({
      endpoint: subscription.endpoint,
      keys: subscription.keys,
      user: req.user._id,
    });

    // Add reference to the User's pushSubscriptions array
    await User.findByIdAndUpdate(
      req.user._id,
      { $addToSet: { pushSubscriptions: newSubscription._id } },
      { new: true }
    );

    res.status(201).json({
      message: "Subscription registered successfully",
      subscription: newSubscription,
    });
  } catch (error) {
    console.error("Error registering push subscription:", error);
    res.status(500).json({ message: "Error registering subscription" });
  }
};

exports.removePushSubscription = async (req, res) => {
  try {
    const { endpoint } = req.body;

    await User.findByIdAndUpdate(
      req.user._id,
      { $pull: { pushSubscriptions: { endpoint } } },
      { new: true }
    );

    res.status(200).json({ message: "Subscription removed successfully" });
  } catch (error) {
    console.error("Error removing push subscription:", error);
    res.status(500).json({ message: "Error removing subscription" });
  }
};

exports.createProductExpiryNotification = async (userId, product) => {
  try {
    const preferences = await UserPreferences.findOne({ user: userId });
    if (!preferences) {
      console.log(`No preferences found for user ${userId}`);
      return null;
    }

    const expiryDate = new Date(product.expireDate).toLocaleDateString();
    const message = `"${product.name}" expires on ${expiryDate}.`;

    // Create notification
    const notification = new Notification({
      user: userId,
      title: "Product Expiry Warning",
      type: "product_expiry",
      message,
      productId: product._id,
      expiryDate: product.expireDate,
    });

    // Save it before emitting
    await notification.save();

    // Socket Notification
    if (typeof sendSingleNotification === "function") {
      sendSingleNotification(userId, notification);
    } else {
      console.warn("sendSingleNotification is not defined or not a function.");
    }

    // Push Notification
    if (preferences?.notificationPreferences?.pushEnabled) {
      await sendPushNotification(userId, {
        title: "⚠️ Expired Product",
        body: message,
      });
    }

    // Email Notification
    if (preferences.notificationPreferences?.emailEnabled) {
      await sendEmailNotification(
        userId,
        productExpiryAlert(product, expiryDate)
      );
    }

    return notification;
  } catch (error) {
    console.error("Error creating product expiry notification:", error);
    throw error;
  }
};
exports.createProductSoonExpiryNotification = async (userId, product) => {
  try {
    const preferences = await UserPreferences.findOne({ user: userId });
    if (!preferences) {
      console.log(`No preferences found for user ${userId}`);
      return null;
    }

    const expiryDate = new Date(product.expireDate).toLocaleDateString();
    const message = `"${product.name}" expires on ${expiryDate}.`;

    // Create notification
    const notification = new Notification({
      user: userId,
      title: "Product Expiry Warning",
      type: "product_expiry",
      message,
      productId: product._id,
      expiryDate: product.expireDate,
    });

    // Save it before emitting
    await notification.save();

    // Socket Notification
    if (typeof sendSingleNotification === "function") {
      sendSingleNotification(userId, notification);
    } else {
      console.warn("sendSingleNotification is not defined or not a function.");
    }

    // Push Notification
    if (preferences?.notificationPreferences?.pushEnabled) {
      await sendPushNotification(userId, {
        title: "⚠️ Expired Product",
        body: message,
      });
    }

    // Email Notification
    if (preferences.notificationPreferences?.emailEnabled) {
      await sendEmailNotification(
        userId,
        productSoonExpiryAlert(product, expiryDate)
      );
    }

    return notification;
  } catch (error) {
    console.error("Error creating product expiry notification:", error);
    throw error;
  }
};

exports.createOutOfStockNotification = async (userId, product) => {
  try {
    const preferences = await UserPreferences.findOne({ user: userId });
    if (!preferences) {
      console.log(`No preferences found for user ${userId}`);
      return null;
    }

    const message = `"${product.name}" is now out of stock. Current stock: ${
      product.quantity || 0
    }.`;

    // Create notification
    const notification = new Notification({
      user: userId,
      title: "Product Out of Stock",
      type: "out_of_stock",
      message,
      productId: product._id,
      stockQuantity: product.stockQuantity,
    });

    // Save it before emitting
    await notification.save();

    // Socket Notification
    if (typeof sendSingleNotification === "function") {
      sendSingleNotification(userId, notification);
    } else {
      console.warn("sendSingleNotification is not defined or not a function.");
    }

    // Push Notification
    if (preferences?.notificationPreferences?.pushEnabled) {
      await sendPushNotification(userId, {
        title: "⚠️ Out of Stock",
        body: message,
      });
    }

    // Email Notification
    if (preferences.notificationPreferences?.emailEnabled) {
      await sendEmailNotification(userId, outOfStockAlert(product));
    }

    return notification;
  } catch (error) {
    console.error("Error creating out-of-stock notification:", error);
    throw error;
  }
};

exports.sendSalaryReminderNotification = async (userId, reminderDetails) => {
  try {
    const preferences = await UserPreferences.findOne({ user: userId });
    if (!preferences) {
      console.log(`No preferences found for user ${userId}`);
      return null;
    }
    // Implement your notification logic here
    const message = `Reminder: Pay salaries today (${reminderDetails.reminderDate}th)`;
    // Example for push notification:
    await Notification.create({
      user: userId,
      type: "salary_reminder",
      message,
      title: "💰 Salary Payment Reminder",
    });

    if (preferences?.notificationPreferences?.pushEnabled) {
      await sendPushNotification(userId, {
        title: "Salary Payment Reminder",
        body: message,
      });
    }
    const templateData = {
      reminderDate: reminderDetails.reminderDate,
      reminderTime: reminderDetails.reminderTime,
      companyName: "Power Sales",
    };

    // ✅ Email Notification
    if (preferences.notificationPreferences?.emailEnabled) {
      await sendEmailNotification(
        userId,
        salaryPaymentReminderTemplate(templateData)
      );
    }
  } catch (error) {
    console.error("Error sending salary reminder:", error);
    throw error;
  }
};
