const Permission = require("../models/permissionModel");
const User = require("../models/userModel");

// Assign or update permissions for a user
exports.assignUserPermissions = async (req, res) => {
  try {
    const { userId, permissions } = req.body;

    if (!userId || typeof permissions !== "object") {
      return res.status(400).json({ message: "Invalid input" });
    }

    // Check if the user exists
    const user = await User.findById(userId);
    if (!user) return res.status(404).json({ message: "User not found" });

    let permissionDoc;

    // If user already has permissions, update them
    if (user.permissions) {
      permissionDoc = await Permission.findByIdAndUpdate(
        user.permissions,
        permissions,
        { new: true }
      );
    } else {
      // Else, create a new permission document and link to user
      permissionDoc = await Permission.create({
        user: userId,
        ...permissions,
      });

      user.permissions = permissionDoc._id;
      await user.save();
    }

    res.status(200).json({
      message: "Permissions assigned successfully",
      permissions: permissionDoc,
    });
  } catch (error) {
    console.error("Assign Permissions Error:", error);
    res.status(500).json({ message: "Server error" });
  }
};

// Get a user's permissions
exports.getUserPermissions = async (req, res) => {
  try {
    const { userId } = req.params;
    if (!userId) {
      return res.status(400).json({ message: "User ID is required" });
    }

    const userExists = await User.exists({ _id: userId });

    if (!userExists) {
      return res.status(404).json({ message: "User not found" });
    }

    const permissions = await Permission.findOne({ user: userId });

    if (!permissions) {
      return res
        .status(404)
        .json({ message: "No permissions assigned to this user" });
    }

    res.status(200).json({
      message: "User permissions fetched successfully",
      permissions,
    });
  } catch (error) {
    console.error("Get User Permissions Error:", error);
    res.status(500).json({ message: "Server error" });
  }
};

// Get a user's login permissions

exports.getLoginUserPermissions = async (req, res) => {
  try {
    const userId = req.user._id;

    if (!userId) {
      return res.status(400).json({ message: "User ID is required" });
    }

    const userExists = await User.exists({ _id: userId });

    if (!userExists) {
      return res.status(404).json({ message: "User not found" });
    }

    const permissions = await Permission.findOne({ user: userId });

    if (!permissions) {
      return res
        .status(404)
        .json({ message: "No permissions assigned to this user" });
    }

    res.status(200).json({
      message: "User permissions fetched successfully",
      permissions,
    });
  } catch (error) {
    console.error("Get User Permissions Error:", error);
    res.status(500).json({ message: "Server error" });
  }
};
