const Product = require("../models/productModel");
const Activity = require("../models/activityTrackerModel");
const Category = require("../models/categoryModel");
const catchAsync = require("../utils/catchAsync");
const TallyCard = require("../models/tallyCardModel");
const AppError = require("../utils/appError");
const { Parser } = require("json2csv");
const { generate10DigitUUID } = require("../helpers/generater");
const Outofstocklimit = require("../models/outofstocklimitModel");
const Outofstockintervals = require("../models/outofstockIntervalsModel");
const slugify = require("slugify");

exports.createProduct = catchAsync(async (req, res, next) => {
  const {
    name,
    quantity,
    costPrice,
    sellingPrice,
    expireDate,
    barcode,
    selectedCategories,
  } = req.body;

  let slug = slugify(name) + generate10DigitUUID();
  const alreadyExist = await Product.findOne({ slug });

  if (alreadyExist) {
    return next(new AppError("Product name already exist", 400));
  }

  const products = await new Product({
    name,
    quantity,
    initialQty: quantity,
    expireDate,
    costPrice,
    slug,
    sellingPrice,
    barcode,
    category: selectedCategories,
    user: req.user._id,
  }).save();
  // Log purchase activity
  const activity = new Activity({
    user: req.user._id,
    activityType: "product_added",
    details: {
      name,
      barcode,
      quantity,
      costPrice,
      sellingPrice,
      expireDate,
    },
  });
  await activity.save();
  createTallyCardEntryForInitialStoke(products, req.user._id);
  res.status(200).send(products);
});

const createTallyCardEntryForInitialStoke = async (products, userId) => {
  try {
    const quantity = Number(products.quantity); // Ensures quantity is a number

    const newTallyEntry = new TallyCard({
      product: products._id,
      inflow: quantity,
      outflow: 0,
      stockBalance: quantity,
      previousQty: quantity,
      description: "Initial Stoke",
      user: userId,
      date: new Date(),
    });

    await newTallyEntry.save();
  } catch (error) {
    console.error("Error creating tally card entry:", error);
    throw error;
  }
};

// get all products
exports.getAllProducts = catchAsync(async (req, res) => {
  const products = await Product.find({})
    .select("+active")
    .sort({ createdAt: -1 })
    .populate("category", "_id name slug");

  res.status(200).send(products);
});

// get all products to sell
exports.getAllProductsToSell = catchAsync(async (req, res) => {
  const stocklimit = await Outofstocklimit.findOne(req.params.outofstocklimit);

  const products = await Product.find({
    active: true,
    quantity: { $gt: stocklimit?.stocklimit },
  })
    .select("+active")
    .populate("category", "_id name slug")
    .populate("user", "_id name");
  res.status(200).send(products);
});

exports.getSingleProduct = catchAsync(async (req, res, next) => {
  const product = await Product.findOne({ slug: req.params.slug })
    .select("+active")
    .populate("category", "_id name slug");
  res.send(product);
});

exports.getSingleProductByID = catchAsync(async (req, res, next) => {
  const { productId } = req.params;
  const product = await Product.findById(productId).select("+active");
  if (!product) {
    return res.status(400).json({ error: "Product not found" });
  }
  return res.status(200).send(product);
});

// products instock
exports.getProductsInstock = catchAsync(async (req, res, next) => {
  const stocklimit = await Outofstocklimit.findOne(req.params.outofstocklimit);
  const products = await Product.find({
    $expr: { $gt: [{ $toDouble: "$quantity" }, stocklimit?.stocklimit] },
  })
    .select("+active")
    .populate("category", "_id name slug")
    .populate("user", "_id name")
    .sort({ createdAt: -1 });

  res.status(200).json(products);
});

// products out of stock
exports.getProductsOutOfStock = catchAsync(async (req, res, next) => {
  const stocklimit = await Outofstocklimit.findOne(req.params.outofstocklimit);

  const products = await Product.find({
    $expr: { $lte: [{ $toDouble: "$quantity" }, stocklimit?.stocklimit] },
  })
    .select("+active")
    .populate("category", "_id name slug")
    .populate("user", "_id name")
    .sort({ createdAt: -1 });
  res.status(200).json(products);
});

// abouttooutofstock
exports.getProductAboutToOutofStock = catchAsync(async (req, res, next) => {
  const intervals = await Outofstockintervals.findOne(
    req.params.stockintervals
  );
  const products = await Product.find({
    $and: [
      { $expr: { $gte: [{ $toDouble: "$quantity" }, intervals.intervalone] } },
      { $expr: { $lte: [{ $toDouble: "$quantity" }, intervals.intervaltwo] } },
    ],
  })
    .select("+active")
    .populate("category", "_id name slug")
    .populate("user", "_id name")
    .sort({ createdAt: -1 });
  res.status(200).json(products);
});

exports.getExpiredProduct = catchAsync(async (req, res, next) => {
  const { timeFrame } = req.query;
  // console.log(timeFrame);
  // Set the future expiration date based on the timeFrame query
  let futureDate;
  const currentDate = new Date();

  switch (timeFrame) {
    case "week":
      futureDate = new Date(currentDate.setDate(currentDate.getDate() + 7));
      break;
    case "month":
      futureDate = new Date(currentDate.setMonth(currentDate.getMonth() + 1));
      break;
    case "3months":
      futureDate = new Date(currentDate.setMonth(currentDate.getMonth() + 3));
      break;
    case "year":
      futureDate = new Date(
        currentDate.setFullYear(currentDate.getFullYear() + 1)
      );
      break;
    default:
      futureDate = null; // Show all products if no valid timeFrame is provided
  }

  const filter = {
    expireDate: { $lte: new Date() },
    // hideExpireDate: { $ne: false },
    // active: { $ne: false },
  };

  if (futureDate) {
    filter.expireDate = { $lte: futureDate, $gte: new Date() };
  }

  const products = await Product.find(filter)
    .select("+active")
    .populate("category", "_id name slug")
    .populate("user", "_id name")
    .sort({ createdAt: -1 });

  res.status(200).send(products);
});

// getAboutToExpire
exports.getProductsAboutToExpire = catchAsync(async (req, res, next) => {
  var date = new Date();
  var date10 = new Date(date.getTime());
  date10.setDate(date10.getDate() + 10);

  const products = await Product.find({
    expireDate: { $lte: new Date(date10), $gte: new Date() },
    hideExpireDate: { $ne: false },
  })
    .select("+active")
    .populate("category", "_id name slug")
    .populate("user", "_id name")
    .sort({ createdAt: -1 });

  res.status(200).send(products);
});

// updateQuantity
exports.updateProductQuantity = catchAsync(async (req, res, next) => {
  let { quantity, currentQty } = req.body;
  const { slug } = req.params;
  const product = await Product.findOne({ slug });
  if (!product) {
    return next(new AppError("Product not found", 404));
  }
  let newQuantity = Number(quantity) + Number(currentQty);

  const updatedProduct = await Product.findOneAndUpdate(
    { slug },
    {
      // ...req.body,
      quantity: newQuantity,
    },
    {
      new: true,
    }
  );

  res.status(200).json(updatedProduct);
});

// update product
exports.updateProduct = catchAsync(async (req, res, next) => {
  const {
    name,
    selectedCategory,
    costPrice,
    quantity,
    initialQty,
    sellingPrice,
    barcode,
  } = req.body;
  const { slug } = req.params;
  const product = await Product.findOne({ slug });
  // Generate new slug if the name is different
  let newSlug = slug;
  if (name !== product.name) {
    newSlug = slugify(name);
    // Check if the new slug already exists
    const existingProduct = await Product.findOne({ slug: newSlug });
    if (existingProduct) {
      return res.status(400).json({ error: "Slug already exists" });
    }
  }

  const updateProduct = await Product.findOneAndUpdate(
    { slug: product.slug },
    {
      slug: newSlug,
      name,
      category: selectedCategory,
      costPrice,
      quantity,
      initialQty,
      sellingPrice,
      barcode,
      ...req.body,
    },
    {
      new: true,
    }
  );
  const activity = new Activity({
    user: req.user._id,
    activityType: "product_updated",
    details: {
      name,
      barcode,
      newQuantity: quantity,
      previousQty: Number(product.quantity),
      openQty: initialQty,
      previousSellingPrice: product.sellingPrice,
      newSellingPrice: sellingPrice,
      previousCostPrice: product.costPrice,
      newCostPrice: costPrice,
    },
  });
  await activity.save();
  if (product.quantity !== quantity) {
    createTallyCardEntryForUpdateProduct(
      product,
      updateProduct.quantity,
      req.user._id
    );
  }

  res.status(200).send(updateProduct);
});

const createTallyCardEntryForUpdateProduct = async (
  product,
  newQuantity,
  userId
) => {
  try {
    const newTallyEntry = new TallyCard({
      product: product._id,
      inflow:
        newQuantity >= Number(product.quantity)
          ? newQuantity - Number(product.quantity)
          : 0,
      outflow:
        newQuantity >= Number(product.quantity)
          ? 0
          : Number(product.quantity) - newQuantity,
      previousQty: Number(product.quantity),
      stockBalance: newQuantity,
      description: "Updated Qty",
      user: userId,
      date: new Date(),
    });
    await newTallyEntry.save();
  } catch (error) {
    console.error("Error creating tally card entry:", error);
    throw error;
  }
};

// update product cost price
exports.updateProductCostPrice = catchAsync(async (req, res, next) => {
  const { costPrice } = req.body;
  const { slug } = req.params;
  const product = await Product.findOne({ slug });
  const updateProduct = await Product.findOneAndUpdate(
    { slug: product.slug },
    {
      costPrice,
    },
    {
      new: true,
    }
  );
  const activity = new Activity({
    user: req.user._id,
    activityType: "product_updated",
    details: {
      name: product.name,
      newQuantity: Number(updateProduct.quantity),
      previousQty: Number(product.quantity),
      openQty: updateProduct.initialQty,
      previousSellingPrice: product.sellingPrice,
      newSellingPrice: updateProduct.sellingPrice,
      previousCostPrice: product.costPrice,
      newCostPrice: Number(costPrice),
    },
  });
  await activity.save();

  res.status(200).send(updateProduct);
});

exports.updateProductSellingPrice = catchAsync(async (req, res, next) => {
  const { sellingPrice } = req.body;
  const { id } = req.params;
  const product = await Product.findById(id);
  const updateProduct = await Product.findByIdAndUpdate(
    id,
    {
      sellingPrice,
    },
    {
      new: true,
    }
  );
  const activity = new Activity({
    user: req.user._id,
    activityType: "product_updated",
    details: {
      name: product.name,
      newQuantity: updateProduct.quantity,
      previousQty: product.quantity,
      openQty: updateProduct.initialQty,
      previousSellingPrice: product.sellingPrice,
      newSellingPrice: Number(sellingPrice),
      previousCostPrice: product.costPrice,
      newCostPrice: updateProduct.costPrice,
    },
  });
  await activity.save();
  res.status(200).send(updateProduct);
});

// delete product
exports.removeProduct = catchAsync(async (req, res, next) => {
  const product = await Product.findById(req.params.id);
  // return console.log(product);
  if (!product) {
    return next(new AppError("Product not found", 404));
  }

  const deleteProduct = await Product.findByIdAndDelete({ _id: product._id });
  const activity = new Activity({
    user: req.user._id,
    activityType: "product_deleted",
    details: {
      productId: deleteProduct._id,
      productName: deleteProduct.name,
      productQuantity: deleteProduct.quantity,
      productSellingPrice: deleteProduct.sellingPrice,
      productCostPrice: deleteProduct.costPrice,
    },
  });
  await activity.save();
  res.status(200).json({ message: "Product Deleted" });
});

exports.exportProductData = catchAsync(async (req, res, next) => {
  const fields = [
    "name",
    "initialQty",
    "quantity",
    "costPrice",
    "sellingPrice",
    "expireDate",
  ];
  const opts = { fields };
  const product = await Product.find({});
  const parser = new Parser(opts);
  const csv = parser.parse(product);
  res.status(200).send(Buffer.from(csv));
});

exports.importProductData = catchAsync(async (req, res, next) => {
  const products = req.body.csvFile.map((product) => {
    // Slugify the product name
    const slug = slugify(product.name, { lower: true });

    // Create a new product object with the slugified name
    return {
      ...product,
      slug: slug,
    };
  });

  const data = await Product.insertMany(products);

  res.status(200).send(data);
});

exports.makeProductInActive = catchAsync(async (req, res, next) => {
  const { slug } = req.params;
  const product = await Product.findOne({ slug }).select("+active");
  if (!product) {
    return next(new AppError("User not found", 404));
  }
  const updateproduct = await Product.findOneAndUpdate(
    { slug: product.slug },
    {
      active: false,
    },
    { new: true }
  ).select("+active");

  // await activity.save();
  res.status(200).send({ ok: true });
});

exports.makeProductActive = catchAsync(async (req, res, next) => {
  const { slug } = req.params;
  const product = await Product.findOne({ slug }).select("+active");

  if (!product) {
    return next(new AppError("User not found", 404));
  }
  const updateproduct = await Product.findOneAndUpdate(
    { slug },
    {
      active: true,
    },
    { new: true }
  ).select("+active");
  // const activity = new Activity({
  //   user: req.user._id,
  //   activityType: "product_active",
  //   details: {
  //     productId: updateproduct._id,
  //     productName: updateproduct.name,
  //     productQuantity: updateproduct.quantity,
  //     productSellingPrice: updateproduct.sellingPrice,
  //     productCostPrice: updateproduct.costPrice,
  //   },
  // });
  // await activity.save();
  res.status(200).send({ ok: true });
});

exports.makeProductInActiveUnActive = catchAsync(async (req, res, next) => {
  const { id } = req.params;

  const product = await Product.findById(id).select("+active");
  if (!product) {
    return next(new AppError("product not found", 404));
  }
  const updateFields = {
    active: !product.active, // Toggle the published status
  };

  const updateproduct = await Product.findByIdAndUpdate(id, updateFields, {
    new: true,
  });
  // const activity = new Activity({
  //   user: req.user._id,
  //   activityType: "product_activeinactive",
  //   details: {
  //     productId: updateproduct._id,
  //     productName: updateproduct.name,
  //     productQuantity: updateproduct.quantity,
  //     productSellingPrice: updateproduct.sellingPrice,
  //     productCostPrice: updateproduct.costPrice,
  //   },
  // });
  // await activity.save();
  res.status(200).send({ flag: true, message: "success" });
});

exports.updateAppQuantity = catchAsync(async (req, res, next) => {
  await Product.updateMany(
    { sellingPrice: 16 },
    { slug: "calmel" },
    { new: true }
  );
  res.status(200).send({ ok: true });
});

exports.getSingleProductsByCategory = catchAsync(async (req, res, next) => {
  const category = await Category.findById(req.params.id);
  if (!category) {
    return next(new AppError("category not found", 404));
  }
  let categoryId = category._id;
  const product = await Product.find({ category: categoryId })
    .populate("category", "_id name")
    .sort({ createdAt: -1 });
  // .select(
  //   "_id name slug category createdAt featuredImage description location"
  // );
  res.status(200).send(product);
});

exports.activeMultipleProduct = catchAsync(async (req, res) => {
  const { products } = req.body;

  for (const product of products) {
    await Product.findByIdAndUpdate(product._id, product).select("+active");
  }

  res.status(200).send({ message: "Field products updated successfully" });
});

exports.activeHideExpireDate = catchAsync(async (req, res) => {
  const { products } = req.body;
  for (const product of products) {
    await Product.findByIdAndUpdate(product._id, product);
  }
  res.status(200).send({ message: "Field products updated successfully" });
});
