const Product = require("../models/productModel");
const TallyCard = require("../models/tallyCardModel");
const PurchaseOrder = require("../models/purchaseOrderModel");
const Activity = require("../models/activityTrackerModel");
const purchasesReturns = require("../models/purchasesReturnsModel");
const StartingBalance = require("../models/startingBalanceModel");
const Purchase = require("../models/purchaseModel");
const catchAsync = require("../utils/catchAsync");
const AppError = require("../utils/appError");
const { generate10DigitUUID } = require("../helpers/generater");
const Suppliers = require("../models/suppliersModel");
const SuppliersPayment = require("../models/suppliersPaymentModel");
const mongoose = require("mongoose");
const { isValidObjectId } = mongoose;

// exports.createProductsPurchase = catchAsync(async (req, res, next) => {
//   let {
//     cart,
//     grandQuantity,
//     supplier,
//     amount,
//     purchaseTime,
//     invoiceNumber,
//     paidAmount,
//   } = req.body;
//   // return console.log(req.body);
//   if (!cart || !grandQuantity || !paidAmount) {
//     return next(new AppError("Something is wrong", 400));
//   }

//   // Validate amount
//   if (isNaN(amount)) {
//     return next(new AppError("Amount must be a valid number", 400));
//   }

//   // Create a new purchase
//   const newPurchase = new Purchase({
//     amount,
//     supplier,
//     purchaseTime,
//     user: req.user._id,
//     grandQuantity,
//     products: cart,
//     invoiceNumber,
//     paidAmount: Number(paidAmount),
//     // Do not set balance here; it will be calculated in the pre-save middleware
//   });

//   // Save new purchase (this will trigger the pre-save middleware)
//   await newPurchase.save();

//   // If payment was made, store it in SuppliersPayment
//   if (Number(paidAmount) > 0) {
//     const newPayment = new SuppliersPayment({
//       purchase: newPurchase._id,
//       amount: Number(paidAmount),
//       notes: `Initial payment for invoice ${invoiceNumber}`,
//     });
//     await newPayment.save();
//   }

//   // Create promises for updating product quantities and creating tally card entries
//   const promises = cart.map(async (item) => {
//     const product = await Product.findById(item._id);
//     if (!product) {
//       throw new Error(`Product with ID ${item._id} not found`);
//     }

//     await Promise.all([
//       updateProductQuantity(item._id, item.count),
//       createTallyCardEntryCreatePurchase(
//         item._id,
//         item.count,
//         product.quantity,
//         req.user._id,
//         purchaseTime,
//         item.purchaseId
//       ),
//     ]);
//   });

//   // Wait for all promises to resolve
//   await newPurchase.save();
//   await Promise.all(promises);

//   // Log purchase activity
//   const activity = new Activity({
//     user: req.user._id,
//     activityType: "purchase",
//     details: {
//       cart,
//       grandQuantity,
//       supplier,
//       amount,
//       purchaseTime,
//       invoiceNumber,
//       paidAmount: Number(paidAmount) || 0,
//       balance: newPurchase.balance, // Use the balance calculated by the middleware
//     },
//   });
//   await activity.save();

//   res.status(200).json({
//     msg: "Purchase success!",
//     newPurchase,
//   });
// });
// // Function to update product quantity
// const updateProductQuantity = async (productId, soldQuantity) => {
//   try {
//     await Product.findOneAndUpdate(
//       { _id: productId },
//       {
//         $inc: { quantity: +Number(soldQuantity) }, // Add quantity by soldQuantity
//       }
//     );
//   } catch (error) {
//     console.error("Error updating product quantity:", error);
//     throw error;
//   }
// };

// // Function to create entry in the tally card
// const createTallyCardEntryCreatePurchase = async (
//   productId,
//   soldQuantity,
//   productQuantity,
//   userId,
//   date,
//   purchaseId
// ) => {
//   try {
//     const newTallyEntry = new TallyCard({
//       product: productId,
//       inflow: Number(soldQuantity),
//       outflow: 0,
//       previousQty: Number(productQuantity),
//       stockBalance: Number(productQuantity) + Number(soldQuantity),
//       description: "Purchases",
//       user: userId,
//       date: date ? date : new Date(),
//       purchaseId,
//     });
//     await newTallyEntry.save();
//   } catch (error) {
//     console.error("Error creating tally card entry:", error);
//     throw error;
//   }
// };

exports.createProductsPurchase = catchAsync(async (req, res, next) => {
  const {
    cart,
    grandQuantity,
    supplier,
    amount,
    purchaseTime,
    invoiceNumber,
    paidAmount,
  } = req.body;

  // Validate required fields
  if (!cart || !grandQuantity || paidAmount === undefined) {
    return next(new AppError("Missing required fields", 400));
  }

  // Validate amount
  if (isNaN(amount)) {
    return next(new AppError("Amount must be a valid number", 400));
  }

  // Start a session for atomic operations
  const session = await mongoose.startSession();
  session.startTransaction();

  try {
    // Create and save purchase document
    const newPurchase = new Purchase({
      amount,
      supplier,
      purchaseTime,
      user: req.user._id,
      grandQuantity,
      products: cart,
      invoiceNumber,
      paidAmount: Number(paidAmount),
    });

    await newPurchase.save({ session });

    // Process payment if any
    if (Number(paidAmount) > 0) {
      await SuppliersPayment.create(
        [
          {
            purchase: newPurchase._id,
            amount: Number(paidAmount),
            notes: `Initial payment for invoice ${invoiceNumber}`,
          },
        ],
        { session }
      );
    }

    // Process each product in cart with atomic operations
    await processCartItems(cart, req.user._id, purchaseTime, session);

    // Log activity
    await Activity.create(
      [
        {
          user: req.user._id,
          activityType: "purchase",
          details: {
            cart,
            grandQuantity,
            supplier,
            amount,
            purchaseTime,
            invoiceNumber,
            paidAmount: Number(paidAmount) || 0,
            balance: newPurchase.balance,
          },
        },
      ],
      { session }
    );

    // Commit transaction if all operations succeed
    await session.commitTransaction();

    res.status(200).json({
      status: "success",
      message: "Purchase completed successfully",
      data: {
        purchase: newPurchase,
      },
    });
  } catch (error) {
    // Abort transaction on error
    await session.abortTransaction();
    return next(new AppError(`Purchase failed: ${error.message}`, 500));
  } finally {
    // End session
    session.endSession();
  }
});

// Helper function to process cart items
const processCartItems = async (cart, userId, purchaseTime, session) => {
  const productUpdates = cart.map(async (item) => {
    // Atomic update of product quantity
    const updatedProduct = await Product.findOneAndUpdate(
      { _id: item._id },
      { $inc: { quantity: +Number(item.count) } },
      { new: true, session }
    );

    if (!updatedProduct) {
      throw new Error(`Product with ID ${item._id} not found`);
    }

    // Create tally card entry
    await TallyCard.create(
      [
        {
          product: item._id,
          inflow: Number(item.count),
          outflow: 0,
          previousQty: updatedProduct.quantity - Number(item.count),
          stockBalance: updatedProduct.quantity,
          description: "Purchases",
          user: userId,
          date: purchaseTime || new Date(),
          purchaseId: item.purchaseId,
        },
      ],
      { session }
    );
  });

  await Promise.all(productUpdates);
};

exports.getAllPurchase = catchAsync(async (req, res, next) => {
  const purchases = await Purchase.find({}).sort({ createdAt: -1 });
  res.send(purchases);
});

exports.getProductsPurchaseByDate = catchAsync(async (req, res, next) => {
  const { supplier } = req.query;
  // console.log(supplier);
  let supplierId = null;
  if (supplier) {
    // Parse supplier as an array and get the first element (assuming only one supplier is intended)
    const suppliers = Array.isArray(supplier) ? supplier : [supplier];
    supplierId = suppliers[0];
  }
  if (req.query.startdate && req.query.enddate) {
    startDate = new Date(req.query.startdate);
    startDate.setHours(0, 0, 0, 0);

    endDate = new Date(req.query.enddate);
    endDate.setHours(23, 59, 59, 999);
  } else {
    // beginning of current day
    var startDate = new Date();
    startDate.setHours(0, 0, 0, 0);

    // end of current day
    var endDate = new Date();
    endDate.setHours(23, 59, 59, 999);
  }
  let query = {
    purchaseTime: { $gte: startDate.toJSON(), $lte: endDate.toJSON() },
  };
  if (supplierId) {
    query.supplier = supplierId;
  }

  const docs = await Purchase.find(query)
    .sort({ purchaseTime: -1 })
    .populate("supplier", "name contact")
    .populate("user", "name");

  var result = {
    purchaseTime: startDate,
  };

  if (docs) {
    var grandQuantity = docs.reduce(function (p, c) {
      return p + c.grandQuantity;
    }, 0);
    result.grandQuantity = grandQuantity;
    res.send({
      result,
      docs,
    });
  }
});

exports.getPurchaseOrderByDate = catchAsync(async (req, res, next) => {
  if (req.query.startdate && req.query.enddate) {
    startDate = new Date(req.query.startdate);
    startDate.setHours(0, 0, 0, 0);

    endDate = new Date(req.query.enddate);
    endDate.setHours(23, 59, 59, 999);
  } else {
    // beginning of current day
    var startDate = new Date();
    startDate.setHours(0, 0, 0, 0);

    // end of current day
    var endDate = new Date();
    endDate.setHours(23, 59, 59, 999);
  }

  const docs = await PurchaseOrder.find({
    purchaseTime: { $gte: startDate.toJSON(), $lte: endDate.toJSON() },
  })
    .sort({ purchaseTime: -1 })
    .populate("supplier", "name contact")
    .populate("user", "name");

  var result = {
    purchaseTime: startDate,
  };

  if (docs) {
    var grandQuantity = docs.reduce(function (p, c) {
      return p + c.grandQuantity;
    }, 0);
    result.grandQuantity = grandQuantity;
    res.send({
      result,
      docs,
    });
  }
});

exports.getPurchaseProductsLimit = catchAsync(async (req, res, next) => {
  const purchase = await Purchase.find({ seller: req.user._id }).limit(1).sort({
    purchaseTime: -1,
  });

  res.status(200).send(purchase);
});

// delete
exports.deletePurchase = catchAsync(async (req, res, next) => {
  const { purchaseId } = req.query;
  const purchase = await Purchase.findById(purchaseId);
  if (!purchase) {
    return next(new AppError("Purchase not found", 404));
  }

  const data = await Purchase.findByIdAndRemove(purchase._id);
  const activity = new Activity({
    user: req.user._id,
    activityType: "delete_purchase",
    details: {
      data,
    },
  });
  await activity.save();
  res.status(200).send({ status: "Success" });
});

// update
exports.updatePurchase = catchAsync(async (req, res, next) => {
  let {
    id,
    purchaseId,
    totalItemCount,
    newQuantity,
    receiptId,
    purchaseReturnReasons,
  } = req.body;
  const { slug } = req.params;
  const product = await Product.findOne({ slug });
  const purchases = await Purchase.findById(receiptId);
  const costPrice = Number(req.body.costPrice);
  const purchase = purchases.products.find(
    (product) => product.purchaseId === purchaseId
  );

  // return console.log(purchases);
  const newUpdate = await Product.findOneAndUpdate(
    { slug },
    { quantity: product.quantity - Number(newQuantity), costPrice },
    {
      new: true,
    }
  );
  createTallyCardEntryUpdatePurchase(
    product._id,
    newQuantity,
    newUpdate.quantity,
    product.quantity,
    req.user._id
  );

  const updatedProduct = await Purchase.updateOne(
    { "products.purchaseId": purchaseId },
    {
      $set: {
        "products.$.count": Number(purchase.count) - Number(newQuantity),
        "products.$.purchasereturnQty": Number(newQuantity),
        "products.$.purchaseReturnReasons": purchaseReturnReasons,
        "products.$.costPrice": Number(costPrice),
      },
    },
    {
      new: true,
    }
  );
  const updatedResult = await Purchase.findOne({
    "products.purchaseId": purchaseId,
  });
  const newAmount = updatedResult.products.reduce(
    (total, product) =>
      total + Number(product.costPrice) * Number(product.count),
    0
  );

  const purchaseEntry = await Purchase.findByIdAndUpdate(
    id,
    {
      grandQuantity: Number(totalItemCount),
      amount: newAmount,
    },
    {
      new: true,
    }
  );

  const { amount } = purchaseEntry.products.reduce(
    (acc, product) => {
      acc.amount += Number(product.costPrice) * Number(product.count);
      return acc;
    },
    { amount: 0 } // Initialize amount property to 0
  );
  purchaseEntry.amount = amount;

  const newPurchaseReturn = new purchasesReturns({
    product: product._id,
    user: req.user._id,
    purchaseReturnReasons,
    newQuantity: Number(product.quantity) - Number(newQuantity),
    purchaseId,
    previousQuantity: Number(purchase.count),
    purchasereturnQty: Number(newQuantity),
    remainingPurchaseReturnQty: Number(purchase.count) - Number(newQuantity),
  });
  await newPurchaseReturn.save();
  // Save the updated document to the database
  res.status(200).json(updatedProduct);
});

exports.getAllProductsWithReasons = async (req, res) => {
  try {
    // Find all purchases where at least one product has reasons data
    const purchases = await Purchase.find({
      "products.purchaseReturnReasons": { $exists: true, $ne: " " },
    });

    // Extract products with reasons data from each purchase
    const productsWithReasons = purchases.reduce((result, purchase) => {
      const products = purchase.products.filter(
        (product) => product.purchaseReturnReasons
      );
      return result.concat(products);
    }, []);

    res.status(200).json(productsWithReasons);
  } catch (error) {
    console.error("Error fetching products with reasons:", error);
    res.status(500).json({ error: "Internal Server Error" });
  }
};

exports.createPurchaseOrder = catchAsync(async (req, res, next) => {
  let { cart, grandQuantity, supplier, amount } = req.body;

  if (!cart || !grandQuantity) {
    return next(new AppError("Some thing is wrong", 400));
  }
  let invoiceID = generate10DigitUUID();
  const newPurchaseOrder = new PurchaseOrder({
    amount,
    supplier,
    grandQuantity,
    products: cart,
    user: req.user._id,
    invoiceID: invoiceID,
  });

  await newPurchaseOrder.save();
  // Log purchase order activity
  const activity = new Activity({
    user: req.user._id,
    activityType: "purchase_order",
    details: {
      cart,
      grandQuantity,
      supplier,
      amount,
      invoiceID: newPurchaseOrder.invoiceID,
    },
  });
  await activity.save();
  res.status(200).json({
    msg: "Purchase success!",
    newPurchase: newPurchaseOrder,
  });
});

exports.updatePurchaseOrder = catchAsync(async (req, res, next) => {
  let { id, totalItemCount, purchaseOrderId, newQuantity } = req.body;
  const updatedProduct = await PurchaseOrder.updateOne(
    { "products.purchaseOrderId": purchaseOrderId },
    {
      $set: {
        "products.$.count": Number(newQuantity),
      },
    },
    {
      new: true,
    }
  );
  const purchaseEntry = await PurchaseOrder.findByIdAndUpdate(
    id,
    {
      grandQuantity: Number(totalItemCount),
    },
    {
      new: true,
    }
  );

  const { amount } = purchaseEntry.products.reduce(
    (acc, product) => {
      acc.amount += Number(product.sellingPrice) * Number(product.count);
      return acc;
    },
    { amount: 0 } // Initialize amount property to 0
  );
  purchaseEntry.amount = amount;
  await PurchaseOrder.findByIdAndUpdate(
    id,
    {
      amount: amount,
    },
    {
      new: true,
    }
  );
  // Save the updated document to the database
  res.status(200).json(updatedProduct);
});

exports.deleteProductInPurchase = catchAsync(async (req, res) => {
  let { purchaseReturnReasons } = req.body;
  const { purchaseId, productId } = req.params;
  try {
    // Find the sales entry with the specified salesId
    const purchaseEntry = await Purchase.findById(purchaseId);
    const product = await Product.findOne({ _id: productId });
    // return console.log(productId);
    // Check if the sales entry exists
    if (!purchaseEntry) {
      return res.status(404).json({ error: "Sales entry not found" });
    }

    // Find the product being deleted from the sales
    const findProduct = purchaseEntry.products.find(
      (product) => String(product._id) === productId
    );

    // Check if the product exists
    if (!findProduct) {
      return res
        .status(404)
        .json({ error: "Product not found in the sales entry" });
    }

    // Update the product count in the Product schema
    // await Product.findByIdAndUpdate(findProduct._id, {
    //   $inc: { quantity: Number(product.quantity) - Number(findProduct.count) },
    // });
    const newUpdate = await Product.findByIdAndUpdate(findProduct._id, {
      $inc: { quantity: -Number(findProduct.count) },
    });

    createTallyCardEntryDeletePurchase(
      findProduct._id,
      newUpdate.quantity,
      findProduct.count,
      req.user._id
    );
    // Pull the product from the products array
    const purchasesEntry = await Purchase.findByIdAndUpdate(
      purchaseId,
      {
        $pull: { products: { _id: productId } },
      },
      { new: true }
    );

    const { amount, grandQuantity } = purchasesEntry.products.reduce(
      (acc, product) => {
        acc.amount += Number(product.costPrice) * Number(product.count);
        acc.grandQuantity += Number(product.count);
        return acc;
      },
      { amount: 0, grandQuantity: 0 }
    );

    // Update the sales entry with the new amount and grandQuantity
    purchasesEntry.amount = amount;
    purchasesEntry.grandQuantity = grandQuantity;
    await purchasesEntry.save();

    // If both amount and grandQuantity are zero, delete the sales entry
    if (amount === 0 && grandQuantity === 0) {
      await Purchase.findByIdAndDelete(purchaseId);
      return res.json({ message: "Purchase entry deleted successfully" });
    }
    const newPurchaseReturn = new purchasesReturns({
      product: productId,
      user: req.user._id,
      purchaseReturnReasons: purchaseReturnReasons
        ? purchaseReturnReasons
        : "All items return",
      newQuantity: Number(newUpdate.quantity) - Number(findProduct.count),
      purchaseId,
      previousQuantity: Number(findProduct.count),
      purchasereturnQty: Number(findProduct.count),
      remainingPurchaseReturnQty:
        Number(newUpdate.quantity) - Number(findProduct.count),
    });
    await newPurchaseReturn.save();
    const activity = new Activity({
      user: req.user._id,
      activityType: "purchase_product_delete",
      details: {
        amount,
        grandQuantity,
        purchaseId: purchaseId,
        productId: productId,
      },
    });
    await activity.save();
    // Send a success response
    res.json({ message: "Product deleted successfully", purchasesEntry });
  } catch (error) {
    console.error("Error deleting product:", error);
    res.status(500).json({ error: "Internal server error" });
  }
});

// Function to create entry in the tally card
const createTallyCardEntryDeletePurchase = async (
  productId,
  productQuantity,
  quantity,
  userId
) => {
  try {
    const newTallyEntry = new TallyCard({
      product: productId,
      inflow: Number(productQuantity) - Number(quantity),
      outflow: Number(quantity),
      previousQty:
        Number(quantity) + (Number(productQuantity) - Number(quantity)),
      stockBalance: Number(productQuantity) - Number(quantity),
      description: "Purchase Return",
      user: userId,
      date: new Date(),
    });
    await newTallyEntry.save();
  } catch (error) {
    console.error("Error creating tally card entry:", error);
    throw error;
  }
};

const createTallyCardEntryUpdatePurchase = async (
  productId,
  newQuantity,
  quantity,
  prevQty,
  userId
) => {
  try {
    // console.log(soldQuantity);
    // const product = await Product.findById(productId);
    // if (!product) {
    //   throw new Error("Product not found");
    // }
    const newTallyEntry = new TallyCard({
      product: productId,
      inflow: 0,
      outflow: Number(newQuantity),
      // previousQty: Number(quantity) + Number(newQuantity),
      previousQty: Number(prevQty),
      stockBalance: Number(quantity),
      description: "Purchase Return",
      user: userId,
      date: new Date(),
    });
    await newTallyEntry.save();
  } catch (error) {
    console.error("Error creating tally card entry:", error);
    throw error;
  }
};

exports.getPurchasesReturnsDataByDate = async (req, res) => {
  const { productId } = req.params;
  const { user } = req.query;
  if (req.query.startDate && req.query.endDate) {
    startDate = new Date(req.query.startDate);
    startDate.setHours(0, 0, 0, 0);
    endDate = new Date(req.query.endDate);
    endDate.setHours(23, 59, 59, 999);
  } else {
    // beginning of current day
    var startDate = new Date();
    startDate.setHours(0, 0, 0, 0);
    // end of current day
    var endDate = new Date();
    endDate.setHours(23, 59, 59, 999);
  }

  try {
    let query = { product: productId };
    if (user) {
      query.user = user;
    }

    if (startDate) {
      query.createdAt = { $gte: startDate.toJSON(), $lte: endDate.toJSON() };
    }
    const result = await purchasesReturns
      .find(query)
      .populate("user", "name")
      .populate("product", "name initialQty quantity")
      .sort({ createdAt: -1 });

    // Calculate totalnewQuantity
    let totalNewQuantity = 0;
    result.forEach((item) => {
      totalNewQuantity += Number(item.newQuantity);
    });

    // Calculate previousQuantity
    let totalPreviousQuantity = 0;
    result.forEach((item) => {
      totalPreviousQuantity += Number(item.previousQuantity);
    });

    // Determine purchasereturnQty as the stock balance of the last document (if any)
    let totalPurchasereturnQty = 0;
    result.forEach((item) => {
      totalPurchasereturnQty += Number(item.purchasereturnQty);
    });

    let totalRemainingPurchaseReturnQty = 0;
    result.forEach((item) => {
      totalRemainingPurchaseReturnQty += Number(
        item.remainingPurchaseReturnQty
      );
    });

    // Return the result along with separately calculated newQuantity, previousQuantity, and salesreturnQty
    res.status(200).json({
      totalNewQuantity,
      totalPreviousQuantity,
      totalPurchasereturnQty,
      totalRemainingPurchaseReturnQty,
      data: result,
    });
  } catch (error) {
    console.error("Error:", error);
    res.status(500).json({ message: error.message || "Internal server error" });
  }
};

exports.updatePurchaseDate = catchAsync(async (req, res, next) => {
  const { purchaseTime } = req.body;
  const { id } = req.params;

  // Find the Purchase record by ID
  const purchase = await Purchase.findById(id);

  if (!purchase) {
    return res.status(404).json({ error: "Purchase record not found" });
  }
  // return console.log(purchase);
  await Purchase.findByIdAndUpdate(
    id,
    {
      purchaseTime,
    },
    {
      new: true,
    }
  );
  // Iterate over each product in the purchase record
  for (const product of purchase.products) {
    // Assuming each product has a purchaseId
    const productPurchaseId = product.purchaseId;

    // Find the product in the TallyCard schema by its purchaseId
    const tallyCardProduct = await TallyCard.findOne({
      purchaseId: productPurchaseId,
    });

    if (tallyCardProduct) {
      // Update the date and time of the product in the TallyCard schema
      tallyCardProduct.date = purchaseTime;
      await tallyCardProduct.save();
    } else {
      console.log(
        `Product with purchaseId ${productPurchaseId} not found in TallyCard.`
      );
      // Handle the case where the product is not found in the TallyCard schema
    }
  }

  res.status(200).json({ message: "PurchaseId date updated successfully" });
});

exports.updatePurchaseProductQty = catchAsync(async (req, res, next) => {
  const { productId, changedQty, purchaseId, productPurchaseId, purchaseDate } =
    req.body;
  // console.log(productId);
  // Find the Purchase record by ID
  const purchase = await Purchase.findById(purchaseId);

  const product = await Product.findOne({ _id: productId });
  // return console.log(product);
  if (!purchase) {
    return res.status(404).json({ error: "Purchase record not found" });
  }
  if (!product) {
    return res.status(404).json({ error: "Product record not found" });
  }

  const updatedPurchaseProduct = await Purchase.updateOne(
    { "products.purchaseId": productPurchaseId },
    {
      $set: {
        "products.$.count": Number(changedQty),
      },
    },
    {
      new: true,
    }
  );
  const updatedProduct = await Product.findByIdAndUpdate(
    product._id,
    {
      quantity: changedQty,
    },
    {
      new: true,
    }
  );
  const updatedResult = await Purchase.findOne({
    "products.purchaseId": productPurchaseId,
  });
  const newAmount = updatedResult.products.reduce(
    (total, product) =>
      total + Number(product.costPrice) * Number(product.count),
    0
  );
  const totalItemCount = updatedResult.products.reduce(
    (total, product) => total + Number(product.count),
    0
  );
  const purchaseEntry = await Purchase.findByIdAndUpdate(
    purchaseId,
    {
      grandQuantity: Number(totalItemCount),
      amount: Number(newAmount),
    },
    {
      new: true,
    }
  );

  createTallyCardEntryUpdatePurchaseQuantity(
    productId,
    changedQty,
    product.quantity,
    req.user._id,
    purchaseDate
  );

  res.status(200).json({ message: "PurchaseId date updated successfully" });
});

const createTallyCardEntryUpdatePurchaseQuantity = async (
  productId,
  newQuantity,
  prevQty,
  userId,
  purchaseDate
) => {
  try {
    const newTallyEntry = new TallyCard({
      product: productId,
      inflow:
        Number(newQuantity) >= Number(prevQty)
          ? Number(newQuantity) - Number(prevQty)
          : 0,
      // outflow: prevQty,
      outflow:
        Number(newQuantity) >= Number(prevQty)
          ? 0
          : Number(prevQty) - Number(newQuantity),
      previousQty: Number(prevQty),
      stockBalance: Number(newQuantity),
      description: "Purchase quantity update",
      user: userId,
      date: purchaseDate ? purchaseDate : new Date(),
    });
    await newTallyEntry.save();
  } catch (error) {
    console.error("Error creating tally card entry:", error);
    throw error;
  }
};

exports.updatePurchaseProductDate = catchAsync(async (req, res, next) => {
  const { productPurchaseId, purchaseDate } = req.body;

  const tally = await TallyCard.findOneAndUpdate(
    { purchaseId: productPurchaseId },
    {
      date: purchaseDate,
    },
    {
      new: true,
    }
  );

  res.status(200).json({ message: "PurchaseId date updated successfully" });
});

exports.getUnpaidPurchasesForSupplier = async (req, res, next) => {
  try {
    const { supplierId } = req.params;

    if (!supplierId) {
      return res.status(400).json({
        success: false,
        message: "Supplier ID is required",
      });
    }

    const supplier = await Suppliers.findById(supplierId);
    if (!supplier) {
      return res.status(404).json({
        success: false,
        message: "Supplier not found",
      });
    }

    // Find all purchases for the supplier
    const purchases = await Purchase.find({ supplier: supplierId }).sort({
      createdAt: -1,
    });

    if (!purchases.length) {
      return res.status(200).json({
        success: true,
        SupplierName: supplier.name,
        grandTotalAmount: 0,
        grandTotalPaidAmount: 0,
        grandRemainingBalance: 0,
        data: [],
        message: "No purchases found for this supplier",
      });
    }

    // Format response to match the required structure
    const formattedPurchases = await Promise.all(
      purchases.map(async (purchase) => {
        // Fetch all payments related to this purchase
        const payments = await SuppliersPayment.find({
          purchase: purchase._id,
        }).sort({ createdAt: -1 });

        return {
          purchaseId: purchase._id,
          invoiceNumber: purchase.invoiceNumber || "N/A",
          totalAmount: purchase.amount || 0,
          paidAmount: purchase.amount - (purchase.balance || 0),
          remainingBalance: purchase.balance || 0,
          createdAt: purchase.createdAt,
          updatedAt: purchase.updatedAt,
          payments: payments.map((payment) => ({
            _id: payment._id,
            amount: payment.amount,
            notes: payment.notes,
            createdAt: payment.createdAt,
            updatedAt: payment.updatedAt,
          })),
        };
      })
    );

    // Filter purchases to include only those with payments or unpaid balances
    const filteredPurchases = formattedPurchases.filter(
      (purchase) =>
        purchase.payments.length > 0 || purchase.remainingBalance > 0
    );

    // Calculate grand totals
    const grandTotalAmount = filteredPurchases.reduce(
      (sum, purchase) => sum + purchase.totalAmount,
      0
    );
    const grandTotalPaidAmount = filteredPurchases.reduce(
      (sum, purchase) => sum + purchase.paidAmount,
      0
    );
    const grandRemainingBalance = filteredPurchases.reduce(
      (sum, purchase) => sum + purchase.remainingBalance,
      0
    );

    res.status(200).json({
      success: true,
      SupplierName: supplier.name,
      grandTotalAmount,
      grandTotalPaidAmount,
      grandRemainingBalance,
      data: filteredPurchases,
    });
  } catch (error) {
    next(error);
  }
};
// Get the total outstanding balance for a supplier
exports.getTotalBalanceForSupplier = catchAsync(async (req, res, next) => {
  const { supplierId } = req.params; // Supplier ID from request params

  if (!supplierId) {
    return next(new AppError("Supplier ID is required", 400));
  }

  // Find all unpaid purchases for the supplier (balance > 0)
  const purchases = await Purchase.find({
    supplier: supplierId,
    balance: { $exists: true, $gt: 0 }, // Only purchases where balance > 0 (still owed)
  });

  // Calculate the total outstanding balance
  const totalBalance = purchases.reduce(
    (sum, purchase) => sum + purchase.balance,
    0
  );

  res.status(200).json(totalBalance);
});

exports.repaySupplierBalance = async (req, res, next) => {
  try {
    const { purchaseId, amount } = req.body;
    if (!purchaseId || !amount) {
      return res.status(400).json({
        success: false,
        message: "Purchase ID and amount are required",
      });
    }

    // Find the purchase
    const purchase = await Purchase.findById(purchaseId);
    if (!purchase) {
      return res
        .status(404)
        .json({ success: false, message: "Purchase not found" });
    }

    // Ensure amount is valid
    if (amount <= 0) {
      return res
        .status(400)
        .json({ success: false, message: "Amount must be greater than zero" });
    }

    if (Number(amount) > purchase.balance) {
      return res
        .status(400)
        .json({ success: false, message: "Amount exceeds remaining balance" });
    }

    // Update the balance
    purchase.paidAmount += Number(amount);
    purchase.balance -= Number(amount);
    await purchase.save();

    // Record the payment
    const newPayment = new SuppliersPayment({
      purchase: purchaseId,
      amount: Number(amount),
      notes: `Payment towards invoice ${purchase.invoiceNumber}`,
    });

    await newPayment.save();

    res.status(200).json(purchase);
  } catch (error) {
    next(error);
  }
};

exports.updateSupplierBalance = async (req, res, next) => {
  try {
    const { purchaseId, paymentId, amount, notes } = req.body;

    // Validate inputs
    if (!purchaseId || !amount || isNaN(amount)) {
      return res.status(400).json({
        success: false,
        message: "Purchase ID and a valid amount are required",
      });
    }

    let paymentAmount = Number(amount);
    if (paymentAmount <= 0) {
      return res.status(400).json({
        success: false,
        message: "Amount must be greater than zero",
      });
    }

    // Validate purchaseId
    if (!isValidObjectId(purchaseId)) {
      return res.status(400).json({
        success: false,
        message: "Invalid Purchase ID",
      });
    }

    // Find the purchase
    const purchase = await Purchase.findById({ _id: purchaseId });
    if (!purchase) {
      return res.status(404).json({
        success: false,
        message: "Purchase not found",
      });
    }

    // Ensure totalAmount, paidAmount, and balance are valid numbers
    if (isNaN(purchase.amount)) {
      return res.status(400).json({
        success: false,
        message: "Invalid total amount in purchase record",
      });
    }

    // Initialize paidAmount and balance if they are missing or invalid
    if (isNaN(purchase.paidAmount)) purchase.paidAmount = 0;
    if (isNaN(purchase.balance)) {
      purchase.balance = purchase.amount - purchase.paidAmount;
    }

    let updatedPayment;
    let oldPaymentAmount = 0;

    if (paymentId) {
      // Validate paymentId
      if (!isValidObjectId(paymentId)) {
        return res.status(400).json({
          success: false,
          message: "Invalid Payment ID",
        });
      }

      // Find the existing payment
      updatedPayment = await SuppliersPayment.findById(paymentId);
      if (!updatedPayment) {
        return res.status(404).json({
          success: false,
          message: "Payment record not found",
        });
      }

      oldPaymentAmount = updatedPayment.amount || 0;

      // Calculate new balance safely
      const newPaidAmount =
        purchase.paidAmount - oldPaymentAmount + paymentAmount;
      const newBalance = purchase.totalAmount - newPaidAmount;

      if (newBalance < 0) {
        return res.status(400).json({
          success: false,
          message: "Amount exceeds total invoice value",
        });
      }

      // Update purchase amounts
      purchase.paidAmount = newPaidAmount;
      purchase.balance = newBalance;

      // Update payment record
      updatedPayment.amount = paymentAmount;
      updatedPayment.notes = notes || updatedPayment.notes;
      updatedPayment.updatedAt = new Date();

      await updatedPayment.save();
    } else {
      // Creating a new payment entry
      if (paymentAmount > purchase.balance) {
        return res.status(400).json({
          success: false,
          message: "Amount exceeds remaining balance",
        });
      }

      updatedPayment = new SuppliersPayment({
        purchase: purchaseId,
        amount: paymentAmount,
        notes: notes || `Payment towards invoice ${purchase.invoiceNumber}`,
      });

      await updatedPayment.save();

      purchase.paidAmount += paymentAmount;
      purchase.balance -= paymentAmount;
    }

    // Save the updated purchase
    await purchase.save();

    res.status(200).json({
      success: true,
      data: {
        payment: updatedPayment,
      },
    });
  } catch (error) {
    console.error("Error updating supplier balance:", error);
    next(error);
  }
};

exports.getPurchasePaymentHistory = async (req, res, next) => {
  try {
    const { purchaseId } = req.params;

    if (!mongoose.Types.ObjectId.isValid(purchaseId)) {
      return res.status(400).json({
        success: false,
        message: "Invalid purchase ID",
      });
    }

    // Find the purchase details
    const purchase = await Purchase.findById(purchaseId).select(
      "_id invoiceNumber amount paidAmount balance createdAt updatedAt"
    );

    if (!purchase) {
      return res.status(404).json({
        success: false,
        message: "No purchase found with this ID",
      });
    }

    // Get all payments related to this purchase
    const paymentHistory = await SuppliersPayment.find({
      purchase: purchaseId,
    })
      .sort({ createdAt: 1 }) // Sort payments in chronological order
      .select("_id amount notes createdAt updatedAt");

    res.status(200).json({
      success: true,
      data: {
        purchaseId: purchase._id,
        invoiceNumber: purchase.invoiceNumber,
        totalAmount: purchase.amount,
        paidAmount: purchase.paidAmount,
        remainingBalance: purchase.balance,
        createdAt: purchase.createdAt,
        updatedAt: purchase.updatedAt,
        payments: paymentHistory,
      },
    });
  } catch (error) {
    next(error);
  }
};

// exports.getProductsPurchaseByDate = catchAsync(async (req, res, next) => {
//   let startDate, endDate;

//   if (req.query.startdate && req.query.enddate) {
//     startDate = new Date(req.query.startdate);
//     startDate.setHours(0, 0, 0, 0);

//     endDate = new Date(req.query.enddate);
//     endDate.setHours(23, 59, 59, 999);
//   } else {
//     // beginning of current day
//     startDate = new Date();
//     startDate.setHours(0, 0, 0, 0);

//     // end of current day
//     endDate = new Date();
//     endDate.setHours(23, 59, 59, 999);
//   }

//   const docs = await Purchase.find({
//     purchaseTime: { $gte: startDate.toJSON(), $lte: endDate.toJSON() },
//   })
//     .sort({ purchaseTime: -1 })
//     .populate("supplier", "name contact")
//     .populate("user", "name");

//   // If no purchases found for the current date, retrieve purchases from the previous day
//   if (docs.length === 0) {
//     const previousDate = new Date(startDate);
//     previousDate.setDate(previousDate.getDate() - 1); // Go back one day
//     const previousDocs = await Purchase.find({
//       purchaseTime: { $lt: startDate.toJSON(), $gte: previousDate.toJSON() },
//     })
//       .sort({ purchaseTime: -1 })
//       .populate("supplier", "name contact")
//       .populate("user", "name");
//     if (previousDocs.length > 0) {
//       const grandQuantity = previousDocs.reduce(function (p, c) {
//         return p + c.grandQuantity;
//       }, 0);
//       const result = {
//         purchaseTime: previousDate,
//         grandQuantity: grandQuantity,
//       };
//       return res.send({
//         result,
//         docs: previousDocs,
//       });
//     }
//   }

//   // If purchases are found for the current date, proceed as usual
//   if (docs.length > 0) {
//     const grandQuantity = docs.reduce(function (p, c) {
//       return p + c.grandQuantity;
//     }, 0);
//     const result = {
//       purchaseTime: startDate,
//       grandQuantity: grandQuantity,
//     };
//     return res.send({
//       result,
//       docs,
//     });
//   }

//   // If no purchases are found for the current date and the previous date, return an empty response
//   res.send({
//     result: null,
//     docs: [],
//   });
// });
