const Report = require("../models/reportsModel");
const Sales = require("../models/salesModel");
const Product = require("../models/productModel");
const catchAsync = require("../utils/catchAsync");
const AppError = require("../utils/appError");
const Expenses = require("../models/expensesModel");
const DebtorsHistory = require("../models/debtorsHistoryModel");

exports.createReport = catchAsync(async (req, res, next) => {
  const {
    totalCost,
    totalSales,
    totalExpenses,
    profit,
    subProfit,
    costStartDate,
    costEndDate,
    salesStartDate,
    salesEndDate,
    expensesStartDate,
    expensesEndDate,
  } = req.body;
  if (
    !totalCost ||
    !totalSales ||
    !totalExpenses ||
    !profit ||
    !subProfit ||
    !costStartDate ||
    !costEndDate ||
    !salesStartDate ||
    !salesEndDate ||
    !expensesStartDate ||
    !expensesEndDate
  ) {
    return next(new AppError("No data selected", 500));
  }

  const data = await new Report({
    totalCost,
    totalSales,
    totalExpenses,
    profit,
    subProfit,
    costStartDate,
    costEndDate,
    salesStartDate,
    salesEndDate,
    expensesStartDate,
    expensesEndDate,
    generatedMonth: salesStartDate,
  }).save();
  res.send(data);
});

exports.getAllReports = catchAsync(async (req, res, next) => {
  const data = await Report.find().sort({ createdAt: -1 });
  res.send(data);
});

// delete
exports.deleteReport = catchAsync(async (req, res, next) => {
  const report = await Report.findById(req.params.id);
  if (!report) {
    return next(new AppError("Report not found", 404));
  }
  const data = await Report.findByIdAndRemove(report._id);
  res.status(200).send({ status: "Success" });
});

exports.getStockValueReport = async (req, res) => {
  try {
    // Retrieve all inventory items
    const inventoryItems = await Product.find();

    // Calculate total inventory value
    let totalInventoryValue = 0;
    inventoryItems.forEach((item) => {
      totalInventoryValue += item.quantity * item.costPrice; // Using cost price for valuation
    });

    // Prepare report data
    const reportData = {
      inventoryItems: inventoryItems.map((item) => ({
        name: item.name,
        quantity: item.quantity,
        costPrice: item.costPrice,
        totalCost: item.quantity * item.costPrice, // Total cost based on quantity and cost price
      })),
      totalInventoryValue: totalInventoryValue,
      valuationMethod: "Cost Price", // Example valuation method
      dateTime: new Date().toISOString(),
    };
    // console.log(reportData);
    // Send the report as JSON response
    res.status(200).json(reportData);
  } catch (err) {
    console.error(err);
    res.status(500).json({ message: "Internal server error" });
  }
};

exports.getStockVarianceReport = async (req, res) => {
  try {
    // Retrieve all inventory items
    const inventoryItems = await Product.find();

    // Calculate total expected inventory value
    let totalExpectedInventoryValue = 0;
    inventoryItems.forEach((item) => {
      totalExpectedInventoryValue += item.quantity * item.sellingPrice; // Using cost price for valuation
    });

    // Calculate total actual inventory value
    let totalActualInventoryValue = 0;
    inventoryItems.forEach((item) => {
      totalActualInventoryValue += item.quantity * item.costPrice; // Using selling price for valuation
    });

    // Calculate variance
    const variance = totalActualInventoryValue - totalExpectedInventoryValue;

    // Prepare report data
    const reportData = {
      totalExpectedInventoryValue: totalExpectedInventoryValue,
      totalActualInventoryValue: totalActualInventoryValue,
      variance: variance,
      dateTime: new Date().toISOString(),
    };
    // console.log(reportData);
    // Send the report as JSON response
    res.status(200).json(reportData);
  } catch (err) {
    console.error(err);
    res.status(500).json({ message: "Internal server error" });
  }
};

exports.getStockVarianceIndividualReport = async (req, res) => {
  try {
    // Retrieve all products
    const products = await Product.find().sort({ createdAt: -1 });

    // Prepare array to store variance data for each product
    const productVariances = [];

    // Calculate variance for each product
    for (const product of products) {
      const expectedValue = product.quantity * product.sellingPrice;
      const actualValue = product.quantity * product.costPrice;
      const varianceAmt = actualValue - expectedValue;
      const sohQty = product.quantity;

      // Store variance data for the product
      const productVariance = {
        productId: product._id,
        productName: product.name,
        expectedValue: expectedValue,
        actualValue: actualValue,
        varianceAmt: varianceAmt,
        sohQty: sohQty,
      };
      productVariances.push(productVariance);
    }

    // Prepare report data
    const reportData = {
      productVariances: productVariances,
      dateTime: new Date().toISOString(),
    };

    // Send the report as JSON response
    res.status(200).json(reportData);
  } catch (err) {
    console.error(err);
    res.status(500).json({ message: "Internal server error" });
  }
};

exports.getDailySales = catchAsync(async (req, res, next) => {
  const { year, month } = req.query;

  const matchStage = {
    $match: {
      $expr: {
        $and: [
          { $eq: [{ $year: "$dateTime" }, parseInt(year, 10)] },
          { $eq: [{ $month: "$dateTime" }, parseInt(month, 10)] },
        ],
      },
    },
  };

  // Aggregation for normal Sales
  const salesResults = await Sales.aggregate([
    matchStage,
    {
      $group: {
        _id: {
          $dateToString: { format: "%Y-%m-%d", date: { $toDate: "$dateTime" } },
        },
        numOfSales: { $sum: 1 },
        totalSales: { $sum: "$grandTotal" },
        quantitySold: { $sum: "$quantitySold" },
      },
    },
    { $sort: { _id: 1 } },
  ]);

  // Aggregation for DebtorsSales
  const debtorsSalesResults = await DebtorsHistory.aggregate([
    matchStage,
    {
      $group: {
        _id: {
          $dateToString: { format: "%Y-%m-%d", date: { $toDate: "$dateTime" } },
        },
        numOfSales: { $sum: 1 },
        totalSales: { $sum: "$grandTotal" },
        quantitySold: { $sum: "$quantitySold" },
      },
    },
    { $sort: { _id: 1 } },
  ]);

  // Combine results from both Sales and DebtorsSales
  const combinedResults = [...salesResults, ...debtorsSalesResults];

  // Merge results by date (_id)
  const mergedResults = combinedResults.reduce((acc, curr) => {
    const existingEntry = acc.find((item) => item._id === curr._id);

    if (existingEntry) {
      existingEntry.numOfSales += curr.numOfSales;
      existingEntry.totalSales += curr.totalSales;
      existingEntry.quantitySold += curr.quantitySold;
    } else {
      acc.push(curr);
    }
    return acc;
  }, []);

  // Sort merged results by date (_id) to ensure correct order
  mergedResults.sort((a, b) => new Date(a._id) - new Date(b._id));

  // Return the merged results
  res.status(200).json(mergedResults);
});

exports.getMonthlySalesByDateRange = catchAsync(async (req, res, next) => {
  const { year, startMonth, endMonth } = req.query;

  // Validate input
  if (!year || !startMonth || !endMonth) {
    return next(
      new AppError(
        "Year, startMonth, and endMonth are required query parameters",
        400
      )
    );
  }

  const matchStage = {
    $match: {
      $expr: {
        $and: [
          { $eq: [{ $year: "$dateTime" }, parseInt(year, 10)] },
          { $gte: [{ $month: "$dateTime" }, parseInt(startMonth, 10)] },
          { $lte: [{ $month: "$dateTime" }, parseInt(endMonth, 10)] },
        ],
      },
    },
  };

  // Aggregation for normal Sales
  const salesResults = await Sales.aggregate([
    matchStage,
    {
      $group: {
        _id: {
          $dateToString: { format: "%Y-%m-%d", date: { $toDate: "$dateTime" } },
        },
        numOfSales: { $sum: 1 },
        totalSales: { $sum: "$grandTotal" },
        quantitySold: { $sum: "$quantitySold" },
      },
    },
    { $sort: { _id: 1 } },
  ]);

  // Aggregation for DebtorsSales
  const debtorsSalesResults = await DebtorsHistory.aggregate([
    matchStage,
    {
      $group: {
        _id: {
          $dateToString: { format: "%Y-%m-%d", date: { $toDate: "$dateTime" } },
        },
        numOfSales: { $sum: 1 },
        totalSales: { $sum: "$grandTotal" },
        quantitySold: { $sum: "$quantitySold" },
      },
    },
    { $sort: { _id: 1 } },
  ]);

  // Combine results from both Sales and DebtorsSales
  const combinedResults = [...salesResults, ...debtorsSalesResults];

  // Merge results by date (_id)
  const mergedResults = combinedResults.reduce((acc, curr) => {
    const existingEntry = acc.find((item) => item._id === curr._id);

    if (existingEntry) {
      existingEntry.numOfSales += curr.numOfSales;
      existingEntry.totalSales += curr.totalSales;
      existingEntry.quantitySold += curr.quantitySold;
    } else {
      acc.push(curr);
    }
    return acc;
  }, []);

  // Sort merged results by date (_id) to ensure correct order
  mergedResults.sort((a, b) => new Date(a._id) - new Date(b._id));

  // Return the merged results
  res.status(200).json(mergedResults);
});

// show all data in months
exports.calculateProfitAndSubProfitMonths = async (req, res, next) => {
  try {
    const year = req.body.year; // Assuming year is provided in the request body

    const monthlyData = [];

    // Loop through each month (from January to December)
    for (let month = 0; month < 12; month++) {
      const salesPipeline = [
        {
          $match: {
            dateTime: {
              $gte: new Date(year, month, 1),
              $lt: new Date(year, month + 1, 1),
            },
          },
        },
        {
          $group: {
            _id: null,
            totalSales: { $sum: "$grandTotal" },
            totalCost: {
              $sum: { $sum: "$products.costPrice" },
            },
          },
        },
      ];

      const expensesPipeline = [
        {
          $match: {
            date: {
              $gte: new Date(year, month, 1),
              $lt: new Date(year, month + 1, 1),
            },
          },
        },
        {
          $group: {
            _id: null,
            totalExpenses: { $sum: "$amount" },
          },
        },
      ];

      const [salesData, expensesData] = await Promise.all([
        Sales.aggregate(salesPipeline),
        Expenses.aggregate(expensesPipeline),
      ]);

      const totalSales = salesData.length > 0 ? salesData[0].totalSales : 0;
      const totalCost = salesData.length > 0 ? salesData[0].totalCost : 0;
      const totalExpenses =
        expensesData.length > 0 ? expensesData[0].totalExpenses : 0;

      const subProfit = totalSales - totalCost;
      const profit = subProfit - totalExpenses;

      // Define month name based on the loop index (0-based)
      const monthName = new Date(year, month, 1).toLocaleString("en-US", {
        month: "long",
      });

      // Push monthly data to the array
      monthlyData.push({
        month: monthName,
        totalCost,
        totalSales,
        totalExpenses,
        subProfit,
        profit,
      });
    }

    // Respond with the array of monthly data
    res.status(200).json(monthlyData);
  } catch (error) {
    next(error);
  }
};

exports.calculateProfitAndSubProfitOldFirst = async (req, res, next) => {
  try {
    const { year } = req.query;
    const now = new Date();
    const selectedYear = parseInt(year, 10) || now.getFullYear();

    const monthlyData = [];

    for (let month = 0; month < 12; month++) {
      const monthStart = new Date(selectedYear, month, 1);
      const monthEnd = new Date(selectedYear, month + 1, 0, 23, 59, 59, 999);

      // Step 1: Get raw sales data for the month
      const salesDocs = await Sales.find({
        dateTime: { $gte: monthStart, $lte: monthEnd },
      });

      const debtorsDocs = await DebtorsHistory.find({
        dateTime: { $gte: monthStart, $lte: monthEnd },
        "paymentHistory.paid_amount": { $exists: true, $gt: 0 },
      });

      const expensesData = await Expenses.aggregate([
        {
          $match: {
            date: { $gte: monthStart, $lte: monthEnd },
          },
        },
        {
          $group: {
            _id: null,
            totalExpenses: { $sum: "$amount" },
          },
        },
      ]);

      let totalSales = 0;
      let totalCost = 0;

      const allSales = [...salesDocs, ...debtorsDocs];

      for (const sale of allSales) {
        totalSales += sale.grandTotal || 0;

        for (const product of sale.products) {
          totalCost += (product.costPrice || 0) * (product.count || 0);
        }
      }

      const totalExpenses = expensesData[0]?.totalExpenses || 0;

      const subProfit = totalSales - totalCost;
      const profit = subProfit - totalExpenses;

      const profitMargin =
        totalSales > 0
          ? parseFloat(((profit / totalSales) * 100).toFixed(2))
          : 0;

      if (totalSales || totalCost || totalExpenses) {
        monthlyData.push({
          month: monthStart.toLocaleString("en-US", { month: "long" }),
          totalSales,
          totalCost,
          totalExpenses,
          subProfit,
          profit,
          profitMargin,
        });
      }
    }

    return res.status(200).json(monthlyData);
  } catch (error) {
    next(error);
  }
};
// only track
exports.calculateProfitAndSubProfitOlderSec = async (req, res, next) => {
  try {
    const { year } = req.query;
    const now = new Date();
    const selectedYear = parseInt(year, 10) || now.getFullYear();

    const monthlyData = [];

    for (let month = 0; month < 12; month++) {
      const monthStart = new Date(selectedYear, month, 1);
      const monthEnd = new Date(selectedYear, month + 1, 0, 23, 59, 59, 999);

      // Step 1: Get raw sales data for the month
      const salesDocs = await Sales.find({
        dateTime: { $gte: monthStart, $lte: monthEnd },
      });

      // Step 2: Get raw debtor data
      const rawDebtorsDocs = await DebtorsHistory.find({
        dateTime: { $gte: monthStart, $lte: monthEnd },
        "paymentHistory.paid_amount": { $exists: true, $gt: 0 },
      });

      // Step 3: Filter fully paid debtors only
      const debtorsDocs = rawDebtorsDocs.filter((doc) => {
        const totalPaid = (doc.paymentHistory || []).reduce(
          (sum, entry) => sum + (entry.paid_amount || 0),
          0
        );
        return totalPaid === doc.grandTotal;
      });

      // Step 4: Get expenses
      const expensesData = await Expenses.aggregate([
        {
          $match: {
            date: { $gte: monthStart, $lte: monthEnd },
          },
        },
        {
          $group: {
            _id: null,
            totalExpenses: { $sum: "$amount" },
          },
        },
      ]);

      // Step 5: Combine and calculate
      let totalSales = 0;
      let totalCost = 0;

      const allSales = [...salesDocs, ...debtorsDocs];

      for (const sale of allSales) {
        totalSales += sale.grandTotal || 0;

        for (const product of sale.products) {
          totalCost += (product.costPrice || 0) * (product.count || 0);
        }
      }

      const totalExpenses = expensesData[0]?.totalExpenses || 0;

      const subProfit = totalSales - totalCost;
      const profit = subProfit - totalExpenses;

      const profitMargin =
        totalSales > 0
          ? parseFloat(((profit / totalSales) * 100).toFixed(2))
          : 0;

      if (totalSales || totalCost || totalExpenses) {
        monthlyData.push({
          month: monthStart.toLocaleString("en-US", { month: "long" }),
          totalSales,
          totalCost,
          totalExpenses,
          subProfit,
          profit,
          profitMargin,
        });
      }
    }

    return res.status(200).json(monthlyData);
  } catch (error) {
    next(error);
  }
};
exports.calculateProfitAndSubProfitNew = async (req, res, next) => {
  try {
    const { year } = req.query;
    const now = new Date();
    const selectedYear = parseInt(year, 10) || now.getFullYear();

    const monthlyData = [];

    for (let month = 0; month < 12; month++) {
      const monthStart = new Date(selectedYear, month, 1);
      const monthEnd = new Date(selectedYear, month + 1, 0, 23, 59, 59, 999);

      // Sales made (cash or credit)
      const salesDocs = await Sales.find({
        dateTime: { $gte: monthStart, $lte: monthEnd },
      });

      // Debtors (credit sales made)
      const debtorsDocs = await DebtorsHistory.find({
        dateTime: { $gte: monthStart, $lte: monthEnd },
      });

      // Payments made by debtors this month
      const debtorPaymentStats = await DebtorsHistory.aggregate([
        { $unwind: "$paymentHistory" },
        {
          $match: {
            "paymentHistory.date_paid": { $gte: monthStart, $lte: monthEnd },
          },
        },
        {
          $group: {
            _id: null,
            totalPaid: { $sum: "$paymentHistory.paid_amount" },
          },
        },
      ]);

      // Expenses this month
      const expensesData = await Expenses.aggregate([
        {
          $match: {
            date: { $gte: monthStart, $lte: monthEnd },
          },
        },
        {
          $group: {
            _id: null,
            totalExpenses: { $sum: "$amount" },
          },
        },
      ]);

      let totalSales = 0;
      let costOfSales = 0;
      let costOfCashSales = 0;
      let amountStillDue = 0;

      const allSales = [...salesDocs, ...debtorsDocs];

      for (const sale of allSales) {
        totalSales += sale.grandTotal || 0;

        for (const product of sale.products || []) {
          costOfSales += (product.costPrice || 0) * (product.count || 0);
        }
        if (sale.paymentMethod === "Credit") {
          const paid = (sale.paymentHistory || []).reduce(
            (sum, entry) => sum + (entry.paid_amount || 0),
            0
          );
          amountStillDue += Math.max((sale.grandTotal || 0) - paid, 0);
        }
      }

      // Calculate actual cash collected
      const cashSales = salesDocs.reduce((sum, doc) => {
        if (["Cash", "MobileMoney", "Gift"].includes(doc.paymentMethod)) {
          return sum + (doc.grandTotal || 0);
        }
        return sum;
      }, 0);

      for (const sale of salesDocs) {
        if (["Cash", "MobileMoney", "Gift"].includes(sale.paymentMethod)) {
          for (const product of sale.products || []) {
            costOfCashSales += (product.costPrice || 0) * (product.count || 0);
          }
        }
      }
      // console.log(cashSales);
      // console.log(costOfCashSales);
      const amountPaidThisMonth = debtorPaymentStats[0]?.totalPaid || 0;
      const cashReceived = cashSales + amountPaidThisMonth;
      const totalExpenses = expensesData[0]?.totalExpenses || 0;

      const grossProfit = totalSales - costOfSales;
      const netProfit = grossProfit - totalExpenses;
      const netCashProfit = cashReceived - costOfSales - totalExpenses;

      const profitMargin =
        totalSales > 0
          ? parseFloat(((netProfit / totalSales) * 100).toFixed(2))
          : 0;

      if (totalSales || cashReceived || costOfSales || totalExpenses) {
        monthlyData.push({
          month: monthStart.toLocaleString("en-US", { month: "long" }),
          totalSales, // All sales (including credit)
          cashReceived, // Actual cash/money collected
          amountPaidThisMonth, // Only from debtors
          amountStillDue, // Credit still unpaid
          cashSales,
          costOfCashSales,
          costOfSales,
          totalExpenses,
          grossProfit,
          netProfit,
          netCashProfit,
          profitMargin,
        });
      }
    }

    return res.status(200).json(monthlyData);
  } catch (error) {
    next(error);
  }
};
exports.calculateProfitAndSubProfit = async (req, res, next) => {
  try {
    const { year } = req.query;
    const now = new Date();
    const selectedYear = parseInt(year, 10) || now.getFullYear();

    const monthlyData = [];

    for (let month = 0; month < 12; month++) {
      const monthStart = new Date(selectedYear, month, 1);
      const monthEnd = new Date(selectedYear, month + 1, 0, 23, 59, 59, 999);

      const salesDocs = await Sales.find({
        dateTime: { $gte: monthStart, $lte: monthEnd },
      });

      const debtorsDocs = await DebtorsHistory.find({
        dateTime: { $gte: monthStart, $lte: monthEnd },
      });

      const debtorPaymentStats = await DebtorsHistory.aggregate([
        { $unwind: "$paymentHistory" },
        {
          $match: {
            "paymentHistory.date_paid": { $gte: monthStart, $lte: monthEnd },
          },
        },
        {
          $group: {
            _id: null,
            totalPaid: { $sum: "$paymentHistory.paid_amount" },
          },
        },
      ]);

      const expensesData = await Expenses.aggregate([
        {
          $match: {
            date: { $gte: monthStart, $lte: monthEnd },
          },
        },
        {
          $group: {
            _id: null,
            totalExpenses: { $sum: "$amount" },
          },
        },
      ]);

      let totalSales = 0;
      let costOfSales = 0;
      let costOfCashSales = 0;
      let costOfDebtorSales = 0; // NEW
      let amountStillDue = 0;

      const allSales = [...salesDocs, ...debtorsDocs];

      for (const sale of allSales) {
        totalSales += sale.grandTotal || 0;

        for (const product of sale.products || []) {
          costOfSales += (product.costPrice || 0) * (product.count || 0);
        }

        if (sale.paymentMethod === "Credit") {
          const paid = (sale.paymentHistory || []).reduce(
            (sum, entry) => sum + (entry.paid_amount || 0),
            0
          );
          amountStillDue += Math.max((sale.grandTotal || 0) - paid, 0);

          // Track debtor product cost
          for (const product of sale.products || []) {
            costOfDebtorSales +=
              (product.costPrice || 0) * (product.count || 0);
          }
        }
      }

      const cashSales = salesDocs.reduce((sum, doc) => {
        if (["Cash", "MobileMoney", "Gift"].includes(doc.paymentMethod)) {
          return sum + (doc.grandTotal || 0);
        }
        return sum;
      }, 0);

      for (const sale of salesDocs) {
        if (["Cash", "MobileMoney", "Gift"].includes(sale.paymentMethod)) {
          for (const product of sale.products || []) {
            costOfCashSales += (product.costPrice || 0) * (product.count || 0);
          }
        }
      }

      const amountPaidThisMonth = debtorPaymentStats[0]?.totalPaid || 0;
      const cashReceived = cashSales + amountPaidThisMonth;
      const totalExpenses = expensesData[0]?.totalExpenses || 0;

      const grossProfit = totalSales - costOfSales;
      const netProfit = grossProfit - totalExpenses;
      const netCashProfit = cashReceived - costOfSales - totalExpenses;

      const profitMargin =
        totalSales > 0
          ? parseFloat(((netProfit / totalSales) * 100).toFixed(2))
          : 0;

      if (totalSales || cashReceived || costOfSales || totalExpenses) {
        monthlyData.push({
          month: monthStart.toLocaleString("en-US", { month: "long" }),
          totalSales,
          cashReceived,
          amountPaidThisMonth,
          amountStillDue,
          cashSales,
          costOfCashSales,
          costOfDebtorSales, // NEW FIELD
          costOfSales,
          totalExpenses,
          grossProfit,
          netProfit,
          netCashProfit,
          profitMargin,
        });
      }
    }

    return res.status(200).json(monthlyData);
  } catch (error) {
    next(error);
  }
};

exports.getGrandTotalsForYearOld = async (req, res, next) => {
  try {
    const { year, period } = req.query;
    const now = new Date();

    let startDate, endDate;

    switch (period) {
      case "today":
        startDate = new Date();
        startDate.setHours(0, 0, 0, 0);
        endDate = new Date();
        endDate.setHours(23, 59, 59, 999);
        break;
      case "yesterday":
        startDate = new Date();
        startDate.setDate(startDate.getDate() - 1);
        startDate.setHours(0, 0, 0, 0);
        endDate = new Date(startDate);
        endDate.setHours(23, 59, 59, 999);
        break;
      case "thisweek":
        const currentDate = new Date(now);
        const dayOfWeek = currentDate.getDay();
        const diffToMonday = dayOfWeek === 0 ? -6 : 1 - dayOfWeek;
        startDate = new Date(currentDate);
        startDate.setDate(currentDate.getDate() + diffToMonday);
        startDate.setHours(0, 0, 0, 0);
        endDate = new Date();
        endDate.setHours(23, 59, 59, 999);
        break;
      case "last30days":
        endDate = new Date();
        endDate.setHours(23, 59, 59, 999);
        startDate = new Date();
        startDate.setDate(startDate.getDate() - 29);
        startDate.setHours(0, 0, 0, 0);
        break;
      default:
        startDate = null;
        endDate = null;
    }

    let grandTotalSales = 0;
    let grandTotalCost = 0;
    let grandTotalExpenses = 0;
    let grandSubProfit = 0;
    let grandProfit = 0;

    const computeTotals = async (start, end) => {
      const [salesDocs, debtorsDocs, expensesData] = await Promise.all([
        Sales.find({
          dateTime: { $gte: start, $lte: end },
          paymentMethod: { $in: ["Cash", "MobileMoney", "Gift"] },
        }),
        DebtorsHistory.find({
          dateTime: { $gte: start, $lte: end },
          "paymentHistory.paid_amount": { $exists: true, $gt: 0 },
        }),
        Expenses.aggregate([
          {
            $match: {
              date: { $gte: start, $lte: end },
            },
          },
          {
            $group: {
              _id: null,
              totalExpenses: { $sum: "$amount" },
            },
          },
        ]),
      ]);

      let totalSales = 0;
      let totalCost = 0;
      console.log("totalCost", totalCost);
      console.log("cashOfSales", cashOfSales);
      for (const sale of salesDocs) {
        totalSales += sale.grandTotal || 0;
        for (const product of sale.products) {
          totalCost += (product.costPrice || 0) * (product.count || 0);
          cashOfSales += (product.sellingPrice || 0) * (product.count || 0);
        }
      }

      for (const debtor of debtorsDocs) {
        // Sum paid amount only
        const paid = debtor.paymentHistory?.reduce(
          (sum, p) => sum + (p.paid_amount || 0),
          0
        );
        totalSales += paid || 0;
        for (const product of debtor.products) {
          totalCost += (product.costPrice || 0) * (product.count || 0);
        }
      }

      const totalExpenses = expensesData[0]?.totalExpenses || 0;
      const subProfit = totalSales - totalCost;
      const profit = subProfit - totalExpenses;

      return { totalSales, totalCost, totalExpenses, subProfit, profit };
    };

    if (startDate && endDate) {
      // Period-based (today, thisweek, etc.)
      const result = await computeTotals(startDate, endDate);

      grandTotalSales = result.totalSales;
      grandTotalCost = result.totalCost;
      grandTotalExpenses = result.totalExpenses;
      grandSubProfit = result.subProfit;
      grandProfit = result.profit;
    } else {
      // Monthly loop (year view)
      const selectedYear = parseInt(year, 10) || now.getFullYear();
      for (let m = 0; m < 12; m++) {
        const start = new Date(selectedYear, m, 1);
        const end = new Date(selectedYear, m + 1, 0, 23, 59, 59, 999);

        const result = await computeTotals(start, end);

        grandTotalSales += result.totalSales;
        grandTotalCost += result.totalCost;
        grandTotalExpenses += result.totalExpenses;
        grandSubProfit += result.subProfit;
        grandProfit += result.profit;
      }
    }

    return res.status(200).json({
      status: "success",
      year: year || now.getFullYear(),
      period: period || "year",
      grandTotalSales,
      grandTotalCost,
      grandSubProfit,
      grandTotalExpenses,
      grandProfit,
      profitMargin:
        grandTotalSales > 0
          ? parseFloat((grandProfit / grandTotalSales) * 100).toFixed(2)
          : "0.00",
    });
  } catch (error) {
    next(error);
  }
};

exports.getGrandTotalsForYear = async (req, res, next) => {
  try {
    const { year, period } = req.query;
    const now = new Date();
    let startDate, endDate;

    switch (period) {
      case "today":
        startDate = new Date();
        startDate.setHours(0, 0, 0, 0);
        endDate = new Date();
        endDate.setHours(23, 59, 59, 999);
        break;
      case "yesterday":
        startDate = new Date();
        startDate.setDate(startDate.getDate() - 1);
        startDate.setHours(0, 0, 0, 0);
        endDate = new Date(startDate);
        endDate.setHours(23, 59, 59, 999);
        break;
      case "thisweek":
        const currentDate = new Date(now);
        const dayOfWeek = currentDate.getDay();
        const diffToMonday = dayOfWeek === 0 ? -6 : 1 - dayOfWeek;
        startDate = new Date(currentDate);
        startDate.setDate(currentDate.getDate() + diffToMonday);
        startDate.setHours(0, 0, 0, 0);
        endDate = new Date();
        endDate.setHours(23, 59, 59, 999);
        break;
      case "last30days":
        endDate = new Date();
        endDate.setHours(23, 59, 59, 999);
        startDate = new Date();
        startDate.setDate(startDate.getDate() - 29);
        startDate.setHours(0, 0, 0, 0);
        break;
      default:
        startDate = null;
        endDate = null;
    }

    // Accumulators
    let totalSales = 0;
    let costOfSales = 0;
    let cashOfSales = 0;
    let totalExpenses = 0;
    let amountStillDue = 0;
    let amountPaidByDebtors = 0;
    let cashSales = 0;

    const computeTotals = async (start, end) => {
      const [salesDocs, debtorsDocs, expensesData, debtorPayments] =
        await Promise.all([
          Sales.find({ dateTime: { $gte: start, $lte: end } }),
          DebtorsHistory.find({ dateTime: { $gte: start, $lte: end } }),
          Expenses.aggregate([
            { $match: { date: { $gte: start, $lte: end } } },
            { $group: { _id: null, total: { $sum: "$amount" } } },
          ]),
          DebtorsHistory.aggregate([
            { $unwind: "$paymentHistory" },
            {
              $match: {
                "paymentHistory.date_paid": { $gte: start, $lte: end },
              },
            },
            {
              $group: {
                _id: null,
                totalPaid: { $sum: "$paymentHistory.paid_amount" },
              },
            },
          ]),
        ]);

      let localTotalSales = 0;
      let localCost = 0;
      let localCashSales = 0;
      let localCashOfSales = 0;
      let localStillDue = 0;

      const allSales = [...salesDocs, ...debtorsDocs];

      for (const sale of allSales) {
        localTotalSales += sale.grandTotal || 0;

        for (const product of sale.products || []) {
          localCost += (product.costPrice || 0) * (product.count || 0);
        }

        if (sale.paymentMethod === "Credit") {
          const paid = (sale.paymentHistory || []).reduce(
            (sum, p) => sum + (p.paid_amount || 0),
            0
          );
          localStillDue += Math.max((sale.grandTotal || 0) - paid, 0);
        }
      }

      localCashSales = salesDocs.reduce((sum, doc) => {
        if (["Cash", "MobileMoney", "Gift"].includes(doc.paymentMethod)) {
          return sum + (doc.grandTotal || 0);
        }
        return sum;
      }, 0);

      // Calculate cashOfSales (cost of goods sold for cash-based sales)
      for (const sale of salesDocs) {
        if (["Cash", "MobileMoney", "Gift"].includes(sale.paymentMethod)) {
          for (const product of sale.products || []) {
            localCashOfSales +=
              (product.sellingPrice || 0) * (product.count || 0);
          }
        }
      }

      const localPaidByDebtors = debtorPayments[0]?.totalPaid || 0;
      const localExpenses = expensesData[0]?.total || 0;

      return {
        localTotalSales,
        localCost,
        localCashSales,
        localCashOfSales,
        localStillDue,
        localPaidByDebtors,
        localExpenses,
      };
    };

    if (startDate && endDate) {
      const result = await computeTotals(startDate, endDate);
      totalSales = result.localTotalSales;
      costOfSales = result.localCost;
      cashSales = result.localCashSales;
      cashOfSales = result.localCashOfSales;
      amountStillDue = result.localStillDue;
      amountPaidByDebtors = result.localPaidByDebtors;
      totalExpenses = result.localExpenses;
    } else {
      const selectedYear = parseInt(year, 10) || now.getFullYear();
      for (let m = 0; m < 12; m++) {
        const start = new Date(selectedYear, m, 1);
        const end = new Date(selectedYear, m + 1, 0, 23, 59, 59, 999);
        const result = await computeTotals(start, end);
        totalSales += result.localTotalSales;
        costOfSales += result.localCost;
        cashSales += result.localCashSales;
        cashOfSales += result.localCashOfSales;
        amountStillDue += result.localStillDue;
        amountPaidByDebtors += result.localPaidByDebtors;
        totalExpenses += result.localExpenses;
      }
    }

    const cashReceived = cashSales + amountPaidByDebtors;
    const grossProfit = totalSales - costOfSales;
    const netProfit = grossProfit - totalExpenses;
    const netCashProfit = cashReceived - costOfSales - totalExpenses;

    return res.status(200).json({
      status: "success",
      year: year || now.getFullYear(),
      period: period || "year",
      totalSales,
      cashReceived,
      amountPaidByDebtors,
      amountStillDue,
      costOfSales,
      cashOfSales,
      totalExpenses,
      grossProfit,
      netProfit,
      netCashProfit,
      profitMargin:
        totalSales > 0
          ? parseFloat(((netProfit / totalSales) * 100).toFixed(2))
          : 0,
    });
  } catch (error) {
    next(error);
  }
};

exports.getExpensesStatistics = catchAsync(async (req, res, next) => {
  const { year, month, period } = req.query;
  const now = new Date();
  let startDate, endDate;

  switch (period) {
    case "today":
      startDate = new Date();
      startDate.setHours(0, 0, 0, 0);
      endDate = new Date();
      endDate.setHours(23, 59, 59, 999);
      break;

    case "yesterday":
      startDate = new Date();
      startDate.setDate(startDate.getDate() - 1);
      startDate.setHours(0, 0, 0, 0);
      endDate = new Date(startDate);
      endDate.setHours(23, 59, 59, 999);
      break;

    case "thisweek":
      const currentDate = new Date(now);
      const dayOfWeek = currentDate.getDay();
      const diffToMonday = dayOfWeek === 0 ? -6 : 1 - dayOfWeek;

      startDate = new Date(currentDate);
      startDate.setDate(currentDate.getDate() + diffToMonday);
      startDate.setHours(0, 0, 0, 0);

      endDate = new Date();
      endDate.setHours(23, 59, 59, 999);
      break;

    case "last30days":
      endDate = new Date();
      endDate.setHours(23, 59, 59, 999);
      startDate = new Date();
      startDate.setDate(startDate.getDate() - 29);
      startDate.setHours(0, 0, 0, 0);
      break;

    default: {
      const specifiedYear = parseInt(year || now.getFullYear(), 10);
      const specifiedMonth = month ? parseInt(month, 10) - 1 : null;

      if (month) {
        startDate = new Date(specifiedYear, specifiedMonth, 1, 0, 0, 0, 0);
        endDate = new Date(
          specifiedYear,
          specifiedMonth + 1,
          0,
          23,
          59,
          59,
          999
        );
      } else {
        startDate = new Date(specifiedYear, 0, 1, 0, 0, 0, 0);
        endDate = new Date(specifiedYear, 11, 31, 23, 59, 59, 999);
      }
    }
  }

  const matchConditions = {
    date: {
      $gte: startDate,
      $lte: endDate,
    },
  };

  const totalAmountResult = await Expenses.aggregate([
    { $match: matchConditions },
    {
      $group: {
        _id: null,
        totalAmount: { $sum: "$amount" },
        count: { $sum: 1 },
      },
    },
  ]);

  const expensesList = await Expenses.find(matchConditions).sort({ date: -1 });

  res.status(200).json({
    period: period || `${month ? `${month}/${year}` : year || "all time"}`,
    totalAmount:
      totalAmountResult.length > 0 ? totalAmountResult[0].totalAmount : 0,
    expenses: expensesList,
  });
});

// @desc    Get all debtors with outstanding balances (amount_remaining > 0)
// @route   GET /api/debtors/outstanding
// @access  Private
exports.getOutstandingDebtors = async (req, res) => {
  try {
    // Find all debtors history records where amount_remaining > 0
    const outstandingDebtors = await DebtorsHistory.find({
      $or: [
        { amount_remaining: { $gt: 0 } },
        {
          $and: [
            { paymentHistory: { $exists: true, $size: 0 } },
            { paymentMethod: "Credit" }, // Only include credit transactions
          ],
        },
      ],
    })
      .populate("debtor", "name phone email") // populate debtor details
      .populate("customer", "name phone email") // populate customer details
      .sort({ updatedAt: -1 }); // sort by most recent/ sort by most recent

    // If you want to group by debtor and calculate total outstanding amounts
    const debtorsSummary = outstandingDebtors.reduce((acc, record) => {
      const debtorId = record.debtor?._id?.toString() || "unknown";

      if (!acc[debtorId]) {
        acc[debtorId] = {
          debtor: record.debtor,
          totalOutstanding: 0,
          invoices: [],
        };
      }

      acc[debtorId].totalOutstanding += record.amount_remaining;
      acc[debtorId].invoices.push({
        invoiceId: record._id,
        invoiceID: record.invoiceID,
        date: record.dateTime,
        grandTotal: record.grandTotal,
        amountPaid: record.grandTotal - record.amount_remaining,
        amountRemaining: record.amount_remaining,
        lastPaymentDate:
          record.paymentHistory.length > 0
            ? record.paymentHistory[record.paymentHistory.length - 1].date_paid
            : null,
      });

      return acc;
    }, {});

    res.status(200).json({
      data: Object.values(debtorsSummary),
    });
  } catch (error) {
    console.error("Error fetching outstanding debtors:", error);
    res.status(500).json({
      success: false,
      message: "Server error while fetching outstanding debtors",
      error: error.message,
    });
  }
};

exports.calculateProfitAndSubProfitForPeriod = async (req, res, next) => {
  try {
    const now = new Date();
    const selectedYear = parseInt(req.query.year, 10) || now.getFullYear();

    // === Time Helpers ===
    const startOfDay = (date) => new Date(date.setHours(0, 0, 0, 0));
    const endOfDay = (date) => new Date(date.setHours(23, 59, 59, 999));

    const startOfWeek = (date) => {
      const d = new Date(date);
      const day = d.getDay();
      const diff = d.getDate() - day + (day === 0 ? -6 : 1); // Monday start
      return startOfDay(new Date(d.setDate(diff)));
    };

    // === Date Ranges with year enforcement ===
    const enforceYear = (startDate, endDate) => {
      const yearStart = new Date(selectedYear, 0, 1);
      const yearEnd = new Date(selectedYear, 11, 31, 23, 59, 59, 999);
      return {
        startDate: startDate < yearStart ? yearStart : startDate,
        endDate: endDate > yearEnd ? yearEnd : endDate,
      };
    };

    const today = enforceYear(startOfDay(new Date()), endOfDay(new Date()));
    const yesterdayDate = new Date(Date.now() - 86400000);
    const yesterday = enforceYear(
      startOfDay(yesterdayDate),
      endOfDay(yesterdayDate)
    );
    const week = enforceYear(startOfWeek(new Date()), endOfDay(new Date()));

    // === Main Calculation Function ===
    const calculateFinancials = async (startDate, endDate) => {
      const salesDocs = await Sales.find({
        dateTime: { $gte: startDate, $lte: endDate },
      });
      const debtorsDocs = await DebtorsHistory.find({
        dateTime: { $gte: startDate, $lte: endDate },
      });

      const debtorPayments = await DebtorsHistory.aggregate([
        { $unwind: "$paymentHistory" },
        {
          $match: {
            "paymentHistory.date_paid": { $gte: startDate, $lte: endDate },
          },
        },
        {
          $group: {
            _id: null,
            totalPaid: { $sum: "$paymentHistory.paid_amount" },
          },
        },
      ]);

      const expensesData = await Expenses.aggregate([
        { $match: { date: { $gte: startDate, $lte: endDate } } },
        { $group: { _id: null, totalExpenses: { $sum: "$amount" } } },
      ]);

      let totalSales = 0;
      let costOfSales = 0;
      let costOfCashSales = 0;
      let costOfDebtorSales = 0;
      let cashSales = 0;
      let amountStillDue = 0;

      const allSales = [...salesDocs, ...debtorsDocs];

      for (const sale of allSales) {
        totalSales += sale.grandTotal || 0;

        for (const product of sale.products || []) {
          costOfSales += (product.costPrice || 0) * (product.count || 0);
        }

        if (sale.paymentMethod === "Credit") {
          const paid = (sale.paymentHistory || []).reduce(
            (sum, entry) => sum + (entry.paid_amount || 0),
            0
          );
          amountStillDue += Math.max((sale.grandTotal || 0) - paid, 0);

          for (const product of sale.products || []) {
            costOfDebtorSales +=
              (product.costPrice || 0) * (product.count || 0);
          }
        }
      }

      for (const sale of salesDocs) {
        if (["Cash", "MobileMoney", "Gift"].includes(sale.paymentMethod)) {
          cashSales += sale.grandTotal || 0;
          for (const product of sale.products || []) {
            costOfCashSales += (product.costPrice || 0) * (product.count || 0);
          }
        }
      }

      const amountPaidByDebtors = debtorPayments[0]?.totalPaid || 0;
      const cashReceived = cashSales + amountPaidByDebtors;
      const totalExpenses = expensesData[0]?.totalExpenses || 0;

      const grossProfit = totalSales - costOfSales;
      const netProfit = grossProfit - totalExpenses;
      const netCashProfit = cashReceived - costOfSales - totalExpenses;

      const profitMargin =
        totalSales > 0
          ? parseFloat(((netProfit / totalSales) * 100).toFixed(2))
          : 0;

      return {
        period: `${startDate.toISOString().split("T")[0]} to ${
          endDate.toISOString().split("T")[0]
        }`,
        totalSales,
        cashReceived,
        amountPaidByDebtors,
        amountStillDue,
        costOfSales,
        cashOfSales: cashSales,
        costOfCashSales,
        costOfDebtorSales,
        totalExpenses,
        grossProfit,
        netProfit,
        netCashProfit,
        profitMargin,
      };
    };

    // === Execute all periods
    const [todayData, yesterdayData, thisWeekData] = await Promise.all([
      calculateFinancials(today.startDate, today.endDate),
      calculateFinancials(yesterday.startDate, yesterday.endDate),
      calculateFinancials(week.startDate, week.endDate),
    ]);

    return res.status(200).json({
      year: selectedYear,
      today: todayData,
      yesterday: yesterdayData,
      thisWeek: thisWeekData,
    });
  } catch (error) {
    next(error);
  }
};
