const Salary = require("../models/salaryModel");
const Expenses = require("../models/expensesModel");
const Activity = require("../models/activityTrackerModel");
const APIFeatures = require("../utils/apiFeatures");
const AppError = require("../utils/appError");
const catchAsync = require("../utils/catchAsync");
const { generate10DigitUUID } = require("../helpers/generater");
const slugify = require("slugify");

// Helper function for formatting currency
const formatCurrency = (amount) => {
  return new Intl.NumberFormat("en-US", {
    style: "currency",
    currency: "GHS",
  }).format(amount);
};

// @desc    Get all salaries
// @route   GET /api/v1/salaries
// @access  Private
// @desc    Get all salaries with year/month filtering
// @route   GET /api/v1/salaries
// @access  Private
exports.getAllSalaries = catchAsync(async (req, res, next) => {
  const { year, month } = req.query;

  // Base query
  let query = Salary.find()
    .sort({ createdAt: -1 })
    .populate("employee", "username name email contactNum");

  // Add year filter if provided
  if (year) {
    const startDate = new Date(`${year}-01-01`);
    const endDate = new Date(`${parseInt(year) + 1}-01-01`);
    query = query.where("date").gte(startDate).lt(endDate);
  }

  // Add month filter if provided (requires year to be specified)
  if (month && year) {
    const startDate = new Date(`${year}-${month.padStart(2, "0")}-01`);
    const endDate = new Date(startDate);
    endDate.setMonth(endDate.getMonth() + 1);
    query = query.where("date").gte(startDate).lt(endDate);
  } else if (month) {
    return next(new AppError("Month filtering requires a year parameter", 400));
  }

  // Execute query
  const salaries = await query;

  // Format response
  const formattedSalaries = salaries.map((salary) => ({
    ...salary._doc,
    formattedAmount: formatCurrency(salary.amount),
  }));

  res.status(200).json(salaries);
});
// @desc    Get single salary
// @route   GET /api/v1/salaries/:id
// @access  Private
exports.getSalary = catchAsync(async (req, res, next) => {
  const salary = await Salary.findById(req.params.id).populate(
    "employee",
    "username name email contactNum"
  );

  if (!salary) {
    return next(new AppError("No salary found with that ID", 404));
  }

  res.status(200).json({
    status: "success",
    data: {
      salary: {
        ...salary._doc,
        formattedAmount: formatCurrency(salary.amount),
      },
    },
  });
});

// @desc    Create new salary
// @route   POST /api/v1/salaries
// @access  Private
exports.createSalary = catchAsync(async (req, res, next) => {
  // Create salary without populate
  const createdSalary = await Salary.create(req.body);

  // Populate employee after creation
  const newSalary = await Salary.findById(createdSalary._id).populate(
    "employee",
    "username name email contactNum"
  );

  // Generate slug for expense
  const slug = slugify(newSalary.paymentMethod) + generate10DigitUUID();

  // Log it in Expenses
  await Expenses.create({
    slug,
    name: "Salary",
    amount: newSalary.amount,
    user: req.user._id,
    date: newSalary.date,
    salary: newSalary._id,
  });

  // Log user activity
  const activitySalary = new Activity({
    user: req.user._id,
    activityType: "salary_added",
    details: {
      name: "Salary",
      amount: newSalary.amount,
      date: newSalary.date,
    },
  });

  const activity = new Activity({
    user: req.user._id,
    activityType: "expenses_added",
    details: {
      name: "Salary",
      amount: newSalary.amount,
      expenseDate: newSalary.date,
    },
  });

  await activitySalary.save();
  await activity.save();

  // Send populated salary in response
  res.status(201).json(newSalary);
});

// @desc    Update salary
// @route   PATCH /api/v1/salaries/:id
// @access  Private
exports.updateSalary = catchAsync(async (req, res, next) => {
  const salary = await Salary.findByIdAndUpdate(req.params.id, req.body, {
    new: true,
    runValidators: true,
  }).populate("employee", "username name email contactNum");
  await Expenses.findOneAndUpdate(
    { salary: req.params.id },
    { amount: salary.amount },
    {
      new: true,
      runValidators: true,
    }
  );
  if (!salary) {
    return next(new AppError("No salary found with that ID", 404));
  }

  res.status(200).json(salary);
});

// @desc    Delete salary
// @route   DELETE /api/v1/salaries/:id
// @access  Private
exports.deleteSalary = catchAsync(async (req, res, next) => {
  const salary = await Salary.findByIdAndDelete(req.params.id);

  if (!salary) {
    return next(new AppError("No salary found with that ID", 404));
  }

  res.status(204).json({
    status: "success",
    data: null,
  });
});

// @desc    Search salaries
// @route   GET /api/v1/salaries/search
// @access  Private
exports.searchSalaries = catchAsync(async (req, res, next) => {
  if (!req.query.q) {
    return next(new AppError("Please provide a search query", 400));
  }

  const salaries = await Salary.search(req.query.q);

  res.status(200).json({
    status: "success",
    results: salaries.length,
    data: {
      salaries: salaries.map((salary) => ({
        ...salary._doc,
        formattedAmount: formatCurrency(salary.amount),
      })),
    },
  });
});

// @desc    Get salary statistics
// @route   GET /api/v1/salaries/stats
// @access  Private
exports.getSalaryStats = catchAsync(async (req, res, next) => {
  const stats = await Salary.aggregate([
    {
      $group: {
        _id: "$paymentMethod",
        count: { $sum: 1 },
        totalAmount: { $sum: "$amount" },
        avgAmount: { $avg: "$amount" },
        minAmount: { $min: "$amount" },
        maxAmount: { $max: "$amount" },
      },
    },
    {
      $addFields: {
        formattedTotal: { $toString: { $round: ["$totalAmount", 2] } },
        formattedAvg: { $toString: { $round: ["$avgAmount", 2] } },
        formattedMin: { $toString: { $round: ["$minAmount", 2] } },
        formattedMax: { $toString: { $round: ["$maxAmount", 2] } },
      },
    },
  ]);

  res.status(200).json(stats);
});

// @desc    Get all salaries for a specific employee
// @route   GET /api/v1/salaries/employee/:employeeId
// @access  Private
exports.getEmployeeSalaries = catchAsync(async (req, res, next) => {
  const { employeeId } = req.params;

  // Validate employeeId format if needed
  if (!employeeId) {
    return next(new AppError("Employee ID is required", 400));
  }

  const salaries = await Salary.find({
    employee: employeeId,
  })
    .sort({ date: -1 })
    .populate("employee", "username name email contactNum");

  if (salaries.length === 0) {
    return res.status(404).json({
      status: "fail",
      message: "No salaries found for this employee",
      salaries: [],
    });
  }

  // Calculate total paid to this employee
  const totalAmount = salaries.reduce((sum, salary) => sum + salary.amount, 0);

  res.status(200).json({
    status: "success",
    results: salaries.length,
    data: {
      totalAmount,
      salaries: salaries.map((salary) => ({
        ...salary._doc,
        formattedAmount: formatCurrency(salary.amount),
      })),
    },
  });
});
