const Sales = require("../models/salesModel");
const TallyCard = require("../models/tallyCardModel");
const Activity = require("../models/activityTrackerModel");
const salesReturns = require("../models/salesReturnsModel");
const Product = require("../models/productModel");
const catchAsync = require("../utils/catchAsync");
const DebtorsHistory = require("../models/debtorsHistoryModel");
const mongoose = require("mongoose");
const { createSaleNotification } = require("./notificationController");
const { generateSalesCSV } = require("../utils/csvGenerator");
const Expenses = require("../models/expensesModel");

exports.createSales = catchAsync(async (req, res, next) => {
  const {
    customer,
    grandTotal,
    quantitySold,
    paymentMethod,
    paidAmount,
    cart,
    subTotal,
    discount,
    currency,
    dateTime,
    invoiceID,
  } = req.body;
  let balanceTotal = (Number(paidAmount) - Number(grandTotal)).toFixed(2);

  const sanitizedDiscount = Object.fromEntries(
    Object.entries(discount).map(([productId, discountValue]) => [
      productId,
      discountValue === null ? 0 : discountValue,
    ])
  );
  const validCustomer =
    customer && mongoose.Types.ObjectId.isValid(customer) ? customer : null;

  const newSales = new Sales({
    customer: validCustomer,
    seller: req.user?._id,
    quantitySold,
    subTotal,
    grandTotal,
    paymentMethod,
    products: cart,
    receivedAmount: Number(paidAmount),
    balance: balanceTotal,
    discount: Object.entries(sanitizedDiscount).map(
      ([productId, discountValue]) => ({
        product: productId,
        discount: Number(discountValue),
      })
    ),
    invoiceID,
    currency,
    dateTime,
  });

  // Create promises for updating product quantities and creating tally card entries
  const promises = cart.map(async (item) => {
    const product = await Product.findById(item?._id);
    if (!product) {
      throw new Error(`Product with ID ${item?._id} not found`);
    }

    await Promise.all([
      updateProductQuantity(item?._id, item.count),
      createTallyCardEntryCreateSales(
        item?._id,
        item.count,
        product.quantity,
        req.user?._id,
        dateTime,
        item.salesId
      ),
    ]);
  });

  // Wait for all promises to resolve

  await newSales.save();
  await Promise.all(promises);
  await createSaleNotification(req.user._id, newSales._id);
  const activity = new Activity({
    user: req.user?._id,
    activityType: "sales",
    details: {
      customer,
      grandTotal,
      quantitySold,
      paymentMethod,
      paidAmount,
      cart,
      subTotal,
      discount,
      invoiceID: newSales.invoiceID,
      currency,
      dateTime,
    },
  });
  await activity.save();

  res.status(200).json({
    msg: "success",
    newSales,
  });
});

// Function to update product quantity
const updateProductQuantity = async (productId, soldQuantity) => {
  const decrement = Math.abs(Number(soldQuantity));
  try {
    await Product.findOneAndUpdate(
      { _id: productId },
      {
        $inc: { quantity: -decrement }, // Decrease quantity by soldQuantity
      }
    );
  } catch (error) {
    console.error("Error updating product quantity:", error);
    throw error;
  }
};

// Function to create entry in the tally card
const createTallyCardEntryCreateSales = async (
  productId,
  soldQuantity,
  productQuantity,
  userId,
  dateTime,
  salesId
) => {
  try {
    const newTallyEntry = new TallyCard({
      product: productId,
      inflow: 0,
      outflow: Number(soldQuantity),
      previousQty: Number(productQuantity),
      stockBalance: Number(productQuantity) - Number(soldQuantity),
      description: "Sales",
      user: userId,
      date: dateTime ? dateTime : new Date(),
      salesId,
    });
    await newTallyEntry.save();
  } catch (error) {
    console.error("Error creating tally card entry:", error);
    throw error;
  }
};

exports.getAllSales = catchAsync(async (req, res, next) => {
  const { productId } = req.params;
  const sales = await Sales.find({});
  res.status(200).send(sales);
});

exports.getSalesByUser = catchAsync(async (req, res, next) => {
  const sales = await Sales.find({ seller: req.user?._id })
    .populate("seller", "id name")
    .sort({
      dateTime: -1,
    });
  res.status(200).send(sales);
});

exports.getSalesByUserLimit = catchAsync(async (req, res, next) => {
  // console.log(req.user?._id)
  const sales = await Sales.find({ seller: req.user?._id })
    .populate("seller", "id name")
    .populate("discount.product", "_id name")
    .limit(1)
    .sort({
      dateTime: -1,
    });
  res.send(sales);
});

// GET total sales for the selected day
exports.totalSalesForSelectedDay = catchAsync(async (req, res, next) => {
  // if date is provided
  const { salesStartDate, salesEndDate } = req.body;
  // console.log(req.params);
  if (salesStartDate && salesEndDate) {
    startDate = new Date(salesStartDate);
    startDate.setHours(0, 0, 0, 0);

    endDate = new Date(salesEndDate);
    endDate.setHours(23, 59, 59, 999);
  } else {
    // beginning of current day
    var startDate = new Date();
    startDate.setHours(0, 0, 0, 0);

    // end of current day
    var endDate = new Date();
    endDate.setHours(23, 59, 59, 999);
  }

  const docs = await Sales.find({
    dateTime: { $gte: startDate.toJSON(), $lte: endDate.toJSON() },
  }).sort({ dateTime: -1 });

  var result = {
    dateTime: startDate,
  };
  if (docs) {
    var grandTotal = docs.reduce(function (p, c) {
      return p + c.grandTotal;
    }, 0.0);
    result.grandTotal = parseFloat(parseFloat(grandTotal).toFixed(2));
    return res.status(200).json(grandTotal);
  } else {
    result.grandTotal = 0;
    res.send(result);
  }
});

// user daily sales
exports.getMyDailySalesByDate = catchAsync(async (req, res, next) => {
  if (req.query.startdate && req.query.enddate) {
    startDate = new Date(req.query.startdate);
    startDate.setHours(0, 0, 0, 0);

    endDate = new Date(req.query.enddate);
    endDate.setHours(23, 59, 59, 999);
  } else {
    // beginning of current day
    var startDate = new Date();
    startDate.setHours(0, 0, 0, 0);

    // end of current day
    var endDate = new Date();
    endDate.setHours(23, 59, 59, 999);
  }

  const sales = await Sales.find({
    seller: req.user?._id,
    dateTime: { $gte: startDate.toJSON(), $lte: endDate.toJSON() },
  })
    .populate("seller", "_id name")
    .populate("products", "_id name")
    .populate({
      path: "discount.product",
      model: "Product", // Assuming your product model is named 'Product'
      select: "_id name", // Select the fields you want from the product model
    })
    .sort({ dateTime: -1 });

  var grandTotal = sales.reduce(function (p, c) {
    return p + c.grandTotal;
  }, 0.0);

  var quantitySold = sales.reduce(function (p, c) {
    return p + c.quantitySold;
  }, 0.0);

  res.send({
    sales,
    grandTotal,
    quantitySold,
  });
});

// GET sales for a particular date
exports.getSalesForaParticulardate = catchAsync(async (req, res, next) => {
  if (req.query.startdate && req.query.enddate) {
    startDate = new Date(req.query.startdate);
    startDate.setHours(0, 0, 0, 0);

    endDate = new Date(req.query.enddate);
    endDate.setHours(23, 59, 59, 999);
  } else {
    // beginning of current day
    var startDate = new Date();
    startDate.setHours(0, 0, 0, 0);
    // end of current day
    var endDate = new Date();
    endDate.setHours(23, 59, 59, 999);
  }
  // const sales = await Sales.find({
  //   seller: req.query.seller,
  //   dateTime: { $gte: startDate.toJSON(), $lte: endDate.toJSON() },
  // })
  //   .populate("seller", "id name")
  //   .sort({ dateTime: -1 });

  const sales = await Sales.find({
    seller: req.query.seller,
    dateTime: { $gte: startDate.toJSON(), $lte: endDate.toJSON() },
  })
    .populate("seller", "_id name")
    .populate("products", "_id name")
    .populate({
      path: "discount.product",
      model: "Product", // Assuming your product model is named 'Product'
      select: "_id name", // Select the fields you want from the product model
    })
    .sort({ dateTime: -1 });

  var grandTotal = sales.reduce(function (p, c) {
    return p + c.grandTotal;
  }, 0.0);
  var quantitySold = sales.reduce(function (p, c) {
    return p + c.quantitySold;
  }, 0.0);
  res.status(200).send(sales);
});

exports.getSalesChartInfo = catchAsync(async (req, res, next) => {
  if (req.query.startdate && req.query.enddate) {
    startDate = new Date(req.query.startdate);
    startDate.setHours(0, 0, 0, 0);

    endDate = new Date(req.query.enddate);
    endDate.setHours(23, 59, 59, 999);
  } else {
    // beginning of current day
    var startDate = new Date();
    startDate.setHours(0, 0, 0, 0);

    // end of current day
    var endDate = new Date();
    endDate.setHours(23, 59, 59, 999);
  }
  Sales.aggregate([
    {
      $project: {
        grandTotal: 1,
        month: { $month: "$dateTime" },
      },
    },
    {
      $group: {
        _id: "$month",
        total: { $sum: { $toDouble: "$grandTotal" } },
      },
    },
  ]).then((documents) => {
    res.status(200).json({
      message: "sales chart details obtaine sucessfully",
      sales: documents,
    });
  });
});
// get users daily sales by admin

exports.adminGetUserDailySalesByDate = catchAsync(async (req, res, next) => {
  const { seller, period, startdate, enddate, year, month } = req.query;
  // console.log(month);
  // Check if seller is provided and ensure it contains only one ID
  const sellerId = seller ? (Array.isArray(seller) ? seller[0] : seller) : null;

  // Initialize default dates (today if not provided)
  let startDate = startdate ? new Date(startdate) : new Date();
  startDate.setHours(0, 0, 0, 0); // Start of the day

  let endDate = enddate ? new Date(enddate) : new Date();
  endDate.setHours(23, 59, 59, 999); // End of the day

  // Set date ranges based on period
  const now = new Date();
  const specifiedYear = parseInt(year) || now.getFullYear();
  const specifiedMonth = month ? parseInt(month) - 1 : now.getMonth();

  if (period === "today") {
    startDate = new Date();
    startDate.setHours(0, 0, 0, 0);
    endDate = new Date();
    endDate.setHours(23, 59, 59, 999);
  } else if (period === "yesterday") {
    startDate = new Date();
    startDate.setDate(startDate.getDate() - 1);
    startDate.setHours(0, 0, 0, 0);
    endDate = new Date();
    endDate.setDate(endDate.getDate() - 1);
    endDate.setHours(23, 59, 59, 999);
  } else if (period === "thisweek") {
    startDate = new Date();
    startDate.setDate(now.getDate() - now.getDay()); // Start of the current week
    startDate.setHours(0, 0, 0, 0);
    endDate = new Date();
    endDate.setDate(startDate.getDate() + 6); // End of the week
    endDate.setHours(23, 59, 59, 999);
  } else if (period === "last30days") {
    startDate = new Date();
    startDate.setDate(now.getDate() - 30); // Start 30 days ago
    startDate.setHours(0, 0, 0, 0);
    endDate = new Date();
    endDate.setHours(23, 59, 59, 999);
  } else if (period === "thismonth") {
    startDate = new Date(specifiedYear, specifiedMonth, 1);
    startDate.setHours(0, 0, 0, 0);
    endDate = new Date(specifiedYear, specifiedMonth + 1, 0);
    endDate.setHours(23, 59, 59, 999);
  } else if (period === "lastmonth") {
    const lastMonth = new Date();
    lastMonth.setMonth(now.getMonth() - 1); // Move to last month
    startDate = new Date(lastMonth.getFullYear(), lastMonth.getMonth(), 1); // First day of last month
    startDate.setHours(0, 0, 0, 0);
    endDate = new Date(lastMonth.getFullYear(), lastMonth.getMonth() + 1, 0); // Last day of last month
    endDate.setHours(23, 59, 59, 999);
  } else if (period === "thisyear") {
    startDate = new Date(specifiedYear, 0, 1); // January 1st of this year
    startDate.setHours(0, 0, 0, 0);
    endDate = new Date(specifiedYear, 11, 31); // December 31st of this year
    endDate.setHours(23, 59, 59, 999);
  } else if (period === "lastyear") {
    const lastYear = specifiedYear - 1;
    startDate = new Date(lastYear, 0, 1);
    startDate.setHours(0, 0, 0, 0);
    endDate = new Date(lastYear, 11, 31);
    endDate.setHours(23, 59, 59, 999);
  }

  // Query setup
  const query = {
    dateTime: { $gte: startDate.toISOString(), $lte: endDate.toISOString() },
  };
  if (sellerId) query.seller = sellerId;

  const sales = await Sales.find(query)
    .populate("seller", "_id name")
    .populate("products", "_id name")
    .populate("customer", "_id name phone_number")
    .populate({
      path: "discount.product",
      model: "Product",
      select: "_id name",
    })
    .sort({ dateTime: -1 });

  // Return an empty array if no sales found and seller is specified
  if (!sales.length && sellerId) return res.status(200).json([]);

  // Calculate total sales and quantity sold
  const grandTotal = sales.reduce((sum, sale) => sum + sale.grandTotal, 0);
  const quantitySold = sales.reduce((sum, sale) => sum + sale.quantitySold, 0);

  // Calculate top 10 best-selling products
  const productSales = {};

  sales.forEach((sale) => {
    sale.products.forEach((product) => {
      if (!productSales[product._id]) {
        productSales[product._id] = {
          name: product.name,
          count: 0,
          salesHistory: [],
        };
      }

      const quantity = Number(product.count);
      productSales[product._id].count += quantity;
      productSales[product._id].salesHistory.push({
        date: sale.dateTime,
        quantitySold: quantity,
        sellingPrice: product.sellingPrice,
        saleTotal: product.sellingPrice * quantity,
      });
    });
  });

  const topProducts = Object.values(productSales)
    .map((product) => {
      product.count = product.salesHistory.reduce(
        (total, history) => total + history.quantitySold,
        0
      );
      return product;
    })
    .sort((a, b) => b.count - a.count)
    .slice(0, 10);

  res.status(200).json({ sales, topProducts, grandTotal, quantitySold });
});

exports.salesReturns = catchAsync(async (req, res, next) => {
  let {
    receiptId,
    salesId,
    newQuantity,
    previousQuantity,
    salesReturnReasons,
    sellingPrice,
  } = req.body;

  const { productId } = req.params;
  const product = await Product.findOne({ _id: productId });
  if (!product) {
    return next(new Error("Product not found"));
  }

  const sales = await Sales.findById(receiptId);
  if (!sales) {
    return next(new Error("Sales record not found"));
  }

  const sale = sales.products.find((product) => product.salesId === salesId);
  if (!sale) {
    return next(new Error("Sales entry not found for the provided salesId"));
  }

  // Ensure newQuantity is defined before proceeding
  newQuantity = Number(newQuantity) || 0;

  const newUpdate = await Product.findOneAndUpdate(
    { slug: product.slug },
    { quantity: Number(product.quantity) + newQuantity, sellingPrice },
    {
      new: true,
    }
  );

  createTallyCardEntryUpdateSales(
    product?._id,
    newQuantity,
    newUpdate.quantity,
    product.quantity,
    req.user?._id
  );

  const updateResult = await Sales.updateOne(
    { "products.salesId": salesId },
    {
      $set: {
        "products.$.count":
          sellingPrice === sale.sellingPrice
            ? Number(sale.count) - Number(newQuantity)
            : Number(sale.count),
        "products.$.salesreturnQty": Number(newQuantity),
        "products.$.previousQuantity": Number(sale.count),
        "products.$.salesReturnReasons": salesReturnReasons,
        "products.$.sellingPrice": sellingPrice,
      },
    },
    {
      new: true,
    }
  );

  // Fetch the updated document
  const updatedProduct = await Sales.findOne({ "products.salesId": salesId });
  if (!updatedProduct) {
    return next(new Error("Updated sales entry not found"));
  }

  const newGrandTotal = updatedProduct.products.reduce(
    (total, product) =>
      total + Number(product.sellingPrice) * Number(product.count),
    0
  );

  let grad = Number(sales.grandTotal) - Number(product.sellingPrice);
  let newQuantitySold = Number(sale.count) - Number(newQuantity);

  await Sales.findByIdAndUpdate(
    receiptId,
    {
      quantitySold:
        sellingPrice === sale.sellingPrice
          ? Number(sales.quantitySold) - Number(newQuantity)
          : Number(sales.quantitySold),
      grandTotal:
        Number(sales.quantitySold) !== Number(sales.quantitySold)
          ? grad
          : newGrandTotal,
    },
    {
      new: true,
    }
  );

  const newSalesReturn = new salesReturns({
    product: productId,
    user: req.user?._id,
    salesReturnReasons,
    newQuantity: Number(sale.count) - Number(newQuantity),
    salesId,
    previousQuantity: Number(sale.count),
    salesreturnQty: Number(newQuantity),
  });
  await newSalesReturn.save();

  const activity = new Activity({
    user: req.user?._id,
    activityType: "update_sales",
    details: {
      receiptId,
      salesId,
      newQuantity,
      previousQuantity,
      salesreturnQty: Number(newQuantity),
      salesReturnReasons,
    },
  });
  await activity.save();

  res.status(200).json(updateResult);
});

const createTallyCardEntryUpdateSales = async (
  productId,
  newQuantity,
  quantity,
  prevQty,
  userId
) => {
  try {
    const newTallyEntry = new TallyCard({
      product: productId,
      inflow: Number(newQuantity),
      outflow: 0,
      previousQty: Number(prevQty),
      stockBalance: Number(quantity),
      description: "Sales Return",
      user: userId,
      date: new Date(),
    });
    await newTallyEntry.save();
  } catch (error) {
    console.error("Error creating tally card entry:", error);
    throw error;
  }
};

exports.getAllSalesProductsWithReasons = async (req, res) => {
  try {
    // Find all sales where at least one product has reasons data
    const sales = await Sales.find({
      "products.salesReturnReasons": { $exists: true, $ne: " " },
    })
      .sort({ updatedAt: -1 })
      .populate("seller", "name");

    // Extract products with reasons data from each sales
    const productsWithReasons = sales.reduce((result, sales) => {
      const products = sales.products.filter(
        (product) => product.salesReturnReasons
      );
      return result.concat(products);
    }, []);

    res.status(200).json(productsWithReasons);
  } catch (error) {
    console.error("Error fetching products with reasons:", error);
    res.status(500).json({ error: "Internal Server Error" });
  }
};

exports.deleteProductInSales = catchAsync(async (req, res) => {
  let { previousQuantity, newQuantity, salesReturnReasons } = req.body;
  const { saleId, productId } = req.params;

  try {
    // Find the sales entry with the specified salesId
    const salessEntry = await Sales.findById(saleId);

    // Check if the sales entry exists
    if (!salessEntry) {
      return res.status(404).json({ error: "Sales entry not found" });
    }

    // Find the product being deleted from the sales
    const findProduct = salessEntry.products.find(
      (product) => String(product?._id) === productId
    );

    // Check if the product exists
    if (!findProduct) {
      return res
        .status(404)
        .json({ error: "Product not found in the sales entry" });
    }

    // Update the product count in the Product schema
    const newUpdate = await Product.findByIdAndUpdate(findProduct?._id, {
      $inc: { quantity: Number(findProduct.count) },
    });
    createTallyCardEntryDeleteSales(
      findProduct?._id,
      newUpdate.quantity,
      findProduct.count,
      req.user?._id
    );
    // Pull the product from the products array
    const salesEntry = await Sales.findByIdAndUpdate(
      saleId,
      {
        $pull: { products: { _id: productId } },
      },
      { new: true }
    );

    // Calculate the grandTotal and quantitySold based on the remaining products
    const { grandTotal, quantitySold } = salesEntry.products.reduce(
      (acc, product) => {
        acc.grandTotal += Number(product.sellingPrice) * Number(product.count);
        acc.quantitySold += Number(product.count);
        return acc;
      },
      { grandTotal: 0, quantitySold: 0 }
    );

    // Update the sales entry with the new grandTotal and quantitySold
    salesEntry.grandTotal = grandTotal;
    salesEntry.quantitySold = quantitySold;
    await salesEntry.save();
    // console.log(newQuantity);
    // console.log(quantitySold);
    // return console.log(quantitySold);
    // If both grandTotal and quantitySold are zero, delete the sales entry
    if (grandTotal === 0 && quantitySold === 0) {
      await Sales.findByIdAndDelete(saleId);
      return res.json({ message: "Sales entry deleted successfully" });
    }
    const newSalesReturn = new salesReturns({
      product: productId,
      user: req.user?._id,
      salesReturnReasons: salesReturnReasons
        ? salesReturnReasons
        : "All items return",
      newQuantity: 0,
      salesId: saleId,
      previousQuantity: Number(newQuantity),
      salesreturnQty: Number(newQuantity),
    });
    await newSalesReturn.save();
    const activity = new Activity({
      user: req.user?._id,
      activityType: "sales_product_delete",
      details: {
        grandTotal,
        quantitySold,
        saleId: saleId,
        productId: productId,
      },
    });
    await activity.save();
    // Send a success response
    res.json({ message: "Product deleted successfully", salesEntry });
  } catch (error) {
    console.error("Error deleting product:", error);
    res.status(500).json({ error: "Internal server error" });
  }
});

const createTallyCardEntryDeleteSales = async (
  productId,
  productQuantity,
  quantity,
  userId
) => {
  try {
    const newTallyEntry = new TallyCard({
      product: productId,
      inflow: Number(productQuantity) + Number(quantity),
      outflow: Number(quantity),
      previousQty: Number(productQuantity),
      stockBalance: Number(productQuantity) + Number(quantity),
      description: "Sales Return",
      user: userId,
      date: new Date(),
    });
    await newTallyEntry.save();
  } catch (error) {
    console.error("Error creating tally card entry:", error);
    throw error;
  }
};

exports.getAllCustomers = async (req, res, next) => {
  try {
    const customers = await Sales.aggregate([
      {
        $match: {
          "customer.name": { $ne: null }, // Exclude documents where customer name is null
          "customer.phoneNumber": { $ne: null }, // Exclude documents where customer phoneNumber is null
        },
      },
      // {
      //   $sort: { "customer.phoneNumber": 1 }, // Sort by phoneNumber
      // },
      {
        $group: {
          _id: "$customer.phoneNumber",
          name: { $first: "$customer.name" },
          phoneNumber: { $first: "$customer.phoneNumber" },
          products: {
            $push: {
              productName: "$products.name",
              purchaseDate: "$dateTime",
              grandTotal: "$grandTotal",
            },
          },
        },
      },
      {
        $project: {
          _id: 0,
          name: 1,
          phoneNumber: 1,
          products: 1,
        },
      },
    ]);

    res.status(200).json(customers);
  } catch (err) {
    res.status(500).json({
      status: "error",
      message: err.message,
    });
  }
};

exports.getSalesReturnsDataByDate = async (req, res) => {
  const { productId } = req.params;
  const { user } = req.query;

  if (req.query.startDate && req.query.endDate) {
    startDate = new Date(req.query.startDate);
    startDate.setHours(0, 0, 0, 0);
    endDate = new Date(req.query.endDate);
    endDate.setHours(23, 59, 59, 999);
  } else {
    // beginning of current day
    var startDate = new Date();
    startDate.setHours(0, 0, 0, 0);
    // end of current day
    var endDate = new Date();
    endDate.setHours(23, 59, 59, 999);
  }

  //   console.log(user, startDate);
  try {
    let query = { product: productId };
    if (user) {
      query.user = user;
    }

    if (startDate) {
      query.createdAt = { $gte: startDate.toJSON(), $lte: endDate.toJSON() };
    }
    const result = await salesReturns
      .find(query)
      .populate("user", "name")
      .populate("product", "name initialQty quantity")
      .sort({ createdAt: -1 });

    // Calculate totalnewQuantity
    let totalNewQuantity = 0;
    result.forEach((item) => {
      totalNewQuantity += Number(item.newQuantity);
    });

    // Calculate previousQuantity
    let totalPreviousQuantity = 0;
    result.forEach((item) => {
      totalPreviousQuantity += item.previousQuantity;
    });

    // Determine salesreturnQty as the stock balance of the last document (if any)
    let totalSalesReturnQty = 0;
    result.forEach((item) => {
      totalSalesReturnQty += item.salesreturnQty;
    });

    // Return the result along with separately calculated newQuantity, previousQuantity, and salesreturnQty
    res.status(200).json({
      totalNewQuantity,
      totalPreviousQuantity,
      totalSalesReturnQty,
      data: result,
    });
  } catch (error) {
    console.error("Error:", error);
    res.status(500).json({ message: error.message || "Internal server error" });
  }
};

exports.updateSalesDate = catchAsync(async (req, res, next) => {
  const { dateTime } = req.body;
  const { id } = req.params;

  // Find the sales record by ID
  const sales = await Sales.findById(id);

  if (!sales) {
    return res.status(404).json({ error: "Sales record not found" });
  }
  await Sales.findByIdAndUpdate(
    id,
    {
      dateTime,
    },
    {
      new: true,
    }
  );
  // Iterate over each product in the sales record
  for (const product of sales.products) {
    // Assuming each product has a salesId
    const productSalesId = product.salesId;

    // Find the product in the TallyCard schema by its salesId
    const tallyCardProduct = await TallyCard.findOne({
      salesId: productSalesId,
    });

    if (tallyCardProduct) {
      // Update the date and time of the product in the TallyCard schema
      tallyCardProduct.date = dateTime;
      await tallyCardProduct.save();
    } else {
      console.log(
        `Product with salesId ${productSalesId} not found in TallyCard.`
      );
      // Handle the case where the product is not found in the TallyCard schema
    }
  }

  res.status(200).json({ message: "Sales date updated successfully" });
});

exports.getProductsAllSales = catchAsync(async (req, res, next) => {
  const { productId } = req.params;

  // Find all sales that contain the specified productId in their products array
  const [sales, debtorSales] = await Promise.all([
    Sales.find({ "products.id": productId }),
    DebtorsHistory.find({ "products.id": productId }),
  ]);

  // Initialize variables to calculate total sales
  let totalSales = 0;
  let totalDiscount = 0;
  let productName = "";
  const allRelatedProducts = [];

  // Helper function to process sales data
  const processSales = (saleList) => {
    saleList.forEach((sale) => {
      sale.products.forEach((product) => {
        if (product.id === productId) {
          const subSales = product.count * product.sellingPrice;
          let discountValue = 0;

          // Check for discounts on the product within the current sale
          sale.discount.forEach((discount) => {
            if (discount.product.toString() === productId.toString()) {
              discountValue += discount.discount; // Accumulate discount value
            }
          });

          allRelatedProducts.push({
            productId: productId,
            productName: product.name,
            quantitySold: product.count,
            sellingPrice: product.sellingPrice,
            subSales: subSales,
            dateTime: sale.dateTime,
            discount: discountValue,
          });
          totalSales += subSales;
          totalDiscount += discountValue;
          productName = product.name;
        }
      });
    });
  };

  // Process sales and debtorSales
  processSales(sales);
  processSales(debtorSales);

  // Sort the allRelatedProducts array by dateTime in descending order
  allRelatedProducts.sort(
    (a, b) => new Date(b.dateTime) - new Date(a.dateTime)
  );

  // Respond with the list of products and the total sales
  res.status(200).json({
    products: allRelatedProducts,
    subTotalSales: totalSales,
    totalSales: totalSales - totalDiscount,
    productName: productName,
    totalDiscount: totalDiscount,
  });
});

exports.adminGetProductSalesStatistics = catchAsync(async (req, res, next) => {
  const { productId } = req.query;

  // Fetch the product if productId is provided
  const productFound = productId ? await Product.findById(productId) : null;

  // Date filtering logic (Assuming you'll include the filtering logic if needed)
  let startDate, endDate;

  // Build the query based on productId (and date range if needed)
  let query = {};
  if (startDate && endDate) {
    query.dateTime = { $gte: startDate.toJSON(), $lte: endDate.toJSON() };
  }
  if (productId) {
    query["products?._id"] = productId;
  }

  // Fetch both Sales and DebtorSales with the same query
  const [sales, debtorSales] = await Promise.all([
    Sales.find(query)
      .populate("seller", "_id name")
      .populate("products", "_id name")
      .populate("customer", "_id name phone_number")
      .populate({
        path: "discount.product",
        model: "Product",
        select: "_id name",
      })
      .sort({ dateTime: -1 }),
    DebtorsHistory.find(query)
      .populate("seller", "_id name")
      .populate("products", "_id name")
      .populate("customer", "_id name phone_number")
      .populate({
        path: "discount.product",
        model: "Product",
        select: "_id name",
      })
      .sort({ dateTime: -1 }),
  ]);

  // Combine sales and debtor sales
  const combinedSales = [...sales, ...debtorSales];

  if (!combinedSales.length) {
    return res.status(200).json({});
  }

  // Calculate sales statistics for products
  const productSales = {};

  combinedSales.forEach((sale) => {
    sale.products.forEach((product) => {
      // Check if product?._id is defined before calling toString()
      if (productId && product?._id && product?._id.toString() !== productId)
        return;

      // Initialize product sales history if not already present
      if (!productSales[product?._id]) {
        productSales[product?._id] = {
          name: product.name,
          count: 0,
          salesHistory: [],
        };
      }

      const quantitySold = Number(product.count);
      productSales[product?._id].count += quantitySold;

      productSales[product?._id].salesHistory.push({
        date: sale.dateTime,
        quantitySold,
        sellingPrice: product.sellingPrice,
        saleTotal: Number(product.sellingPrice) * quantitySold,
      });
    });
  });

  // Only return the specific product statistics if productId is provided
  const productHistory =
    productId && productSales[productId]
      ? {
          ...productSales[productId],
          availableQuantity: productFound ? Number(productFound.quantity) : 0, // Add available quantity from productFound
        }
      : null;

  res.status(200).send(productHistory || {});
});

exports.adminGetProductSalesStatistics = catchAsync(async (req, res, next) => {
  const { productId } = req.query;

  // Fetch the product if productId is provided
  const productFound = productId ? await Product.findById(productId) : null;

  // Date filtering logic (Assuming you'll include the filtering logic if needed)
  let startDate, endDate;

  // Build the query based on productId (and date range if needed)
  let query = {};
  if (startDate && endDate) {
    query.dateTime = { $gte: startDate.toJSON(), $lte: endDate.toJSON() };
  }
  if (productId) {
    query["products?._id"] = productId;
  }

  // Fetch both Sales and DebtorSales with the same query
  const [sales, debtorSales] = await Promise.all([
    Sales.find(query)
      .populate("seller", "_id name")
      .populate("products", "_id name")
      .populate("customer", "_id name phone_number")
      .populate({
        path: "discount.product",
        model: "Product",
        select: "_id name",
      }),
    DebtorsHistory.find(query)
      .populate("seller", "_id name")
      .populate("products", "_id name")
      .populate("customer", "_id name phone_number")
      .populate({
        path: "discount.product",
        model: "Product",
        select: "_id name",
      }),
  ]);

  // Combine sales and debtor sales and sort by dateTime
  const combinedSales = [...sales, ...debtorSales].sort(
    (a, b) => new Date(b.dateTime) - new Date(a.dateTime)
  );

  if (!combinedSales.length) {
    return res.status(200).json({});
  }

  // Calculate sales statistics for products
  const productSales = {};

  combinedSales.forEach((sale) => {
    sale.products.forEach((product) => {
      // Check if product?._id is defined before calling toString()
      if (productId && product?._id && product?._id.toString() !== productId)
        return;

      // Initialize product sales history if not already present
      if (!productSales[product?._id]) {
        productSales[product?._id] = {
          name: product.name,
          count: 0,
          salesHistory: [],
        };
      }

      const quantitySold = Number(product.count);
      productSales[product?._id].count += quantitySold;

      productSales[product?._id].salesHistory.push({
        date: sale.dateTime,
        quantitySold,
        sellingPrice: Number(product.sellingPrice),
        saleTotal: Number(product.sellingPrice) * Number(quantitySold),
      });
    });
  });

  // Only return the specific product statistics if productId is provided
  const productHistory =
    productId && productSales[productId]
      ? {
          ...productSales[productId],
          availableQuantity: productFound ? Number(productFound.quantity) : 0, // Add available quantity from productFound
        }
      : null;

  res.status(200).send(productHistory || {});
});

exports.getAllGrandTotalSalesForYear = catchAsync(async (req, res, next) => {
  const { year } = req.query;

  // Default to the current year if no year is specified
  const specifiedYear = parseInt(year, 10) || new Date().getFullYear();

  // Fetch total sales, total product count, and count of sales records
  const salesResults = await Sales.aggregate([
    {
      $match: {
        dateTime: {
          $gte: new Date(`${specifiedYear}-01-01T00:00:00.000Z`),
          $lte: new Date(`${specifiedYear}-12-31T23:59:59.999Z`),
        },
      },
    },
    {
      $group: {
        _id: null,
        totalSales: { $sum: "$grandTotal" }, // Aggregate before unwinding products
        totalSalesCount: { $sum: 1 },
      },
    },
  ]);

  // Fetch total paid amounts from DebtorsHistory
  const debtorsPaidAmountResults = await DebtorsHistory.aggregate([
    {
      $match: {
        dateTime: {
          $gte: new Date(`${specifiedYear}-01-01T00:00:00.000Z`),
          $lte: new Date(`${specifiedYear}-12-31T23:59:59.999Z`),
        },
        "paymentHistory.paid_amount": { $exists: true, $gt: 0 }, // Filter records with paid_amount
      },
    },
    {
      $unwind: "$paymentHistory", // Unwind the paymentHistory array
    },
    {
      $match: {
        "paymentHistory.paid_amount": { $gt: 0 }, // Ensure paid_amount is greater than 0
      },
    },
    {
      $group: {
        _id: null,
        totalPaidAmount: { $sum: "$paymentHistory.paid_amount" }, // Sum up the paid amounts
      },
    },
  ]);

  // Fetch total product count from Sales and DebtorsHistory separately
  const productCounts = await Promise.all([
    Sales.aggregate([
      {
        $match: {
          dateTime: {
            $gte: new Date(`${specifiedYear}-01-01T00:00:00.000Z`),
            $lte: new Date(`${specifiedYear}-12-31T23:59:59.999Z`),
          },
        },
      },
      { $unwind: "$products" },
      { $group: { _id: null, totalProductCount: { $sum: "$products.count" } } },
    ]),
    DebtorsHistory.aggregate([
      {
        $match: {
          dateTime: {
            $gte: new Date(`${specifiedYear}-01-01T00:00:00.000Z`),
            $lte: new Date(`${specifiedYear}-12-31T23:59:59.999Z`),
          },
        },
      },
      { $unwind: "$products" },
      { $group: { _id: null, totalProductCount: { $sum: "$products.count" } } },
    ]),
  ]);

  // Calculate total sales (Sales + Paid Amount from DebtorsHistory)
  const totalSales =
    (salesResults[0]?.totalSales || 0) +
    (debtorsPaidAmountResults[0]?.totalPaidAmount || 0);

  // Calculate total product count (Sales + DebtorsHistory)
  const totalProductCount =
    (productCounts[0][0]?.totalProductCount || 0) +
    (productCounts[1][0]?.totalProductCount || 0);

  res.status(200).json({
    status: "success",
    year: specifiedYear,
    totalSales,
    totalProductCount, // Total count of products sold
  });
});

exports.getSalesAnalyticsByYearMonthOld = catchAsync(async (req, res, next) => {
  const { year, month, period } = req.query;
  const specifiedYear = parseInt(year, 10) || new Date().getFullYear();
  console.log(period);
  // Date range calculation
  const [startDate, endDate] = month
    ? [
        new Date(Date.UTC(specifiedYear, parseInt(month, 10) - 1, 1)),
        new Date(
          Date.UTC(specifiedYear, parseInt(month, 10), 0, 23, 59, 59, 999)
        ),
      ]
    : [
        new Date(Date.UTC(specifiedYear, 0, 1)),
        new Date(Date.UTC(specifiedYear, 11, 31, 23, 59, 59, 999)),
      ];

  const dateFilter = {
    dateTime: { $gte: startDate, $lte: endDate },
  };

  // Main aggregation queries
  const [
    salesResults,
    debtorsResults,
    productStats,
    expensesResults,
    creditSalesHistory,
  ] = await Promise.all([
    // Sales aggregation
    Sales.aggregate([
      { $match: dateFilter },
      { $unwind: "$products" },
      {
        $group: {
          _id: null,
          totalCashSales: {
            $sum: {
              $cond: [{ $eq: ["$paymentMethod", "Cash"] }, "$grandTotal", 0],
            },
          },
          totalMobileMoneySales: {
            $sum: {
              $cond: [
                { $eq: ["$paymentMethod", "MobileMoney"] },
                "$grandTotal",
                0,
              ],
            },
          },
          totalGiftSales: {
            $sum: {
              $cond: [{ $eq: ["$paymentMethod", "Gift"] }, "$grandTotal", 0],
            },
          },
          totalSalesCount: { $sum: 1 },
          totalRevenue: { $sum: "$grandTotal" },
          totalCost: {
            $sum: { $multiply: ["$products.count", "$products.costPrice"] },
          },
          totalQuantitySold: { $sum: "$products.count" },
        },
      },
    ]),

    // Debtors payment history
    DebtorsHistory.aggregate([
      {
        $match: {
          ...dateFilter,
          "paymentHistory.paid_amount": { $exists: true, $gt: 0 },
        },
      },
      { $unwind: "$products" },
      {
        $group: {
          _id: null,
          totalCreditSales: { $sum: "$grandTotal" },
          totalCreditCost: {
            $sum: { $multiply: ["$products.count", "$products.costPrice"] },
          },
          totalPaidAmount: { $sum: { $sum: "$paymentHistory.paid_amount" } },
          outstandingDebt: {
            $sum: {
              $subtract: [
                "$grandTotal",
                { $sum: "$paymentHistory.paid_amount" },
              ],
            },
          },
          creditPaymentCount: { $sum: { $size: "$paymentHistory" } },
        },
      },
    ]),

    // Product statistics
    Sales.aggregate([
      { $match: dateFilter },
      { $unwind: "$products" },
      { $group: { _id: null, totalProductCount: { $sum: "$products.count" } } },
    ]),
    // Expenses
    Expenses.aggregate([
      { $match: { date: { $gte: startDate, $lte: endDate } } },
      {
        $group: {
          _id: null,
          totalExpenses: { $sum: "$amount" },
          expenseCount: { $sum: 1 },
        },
      },
    ]),
    // Outstanding credit
    DebtorsHistory.aggregate([
      {
        $match: {
          paymentMethod: "Credit",
          "paymentHistory.paid_amount": { $exists: true },
          ...dateFilter, // Include the date filter here
        },
      },
      {
        $addFields: {
          totalPaid: { $sum: "$paymentHistory.paid_amount" },
          outstandingBalance: {
            $subtract: ["$grandTotal", { $sum: "$paymentHistory.paid_amount" }],
          },
        },
      },
      {
        $match: {
          outstandingBalance: { $gt: 0 },
        },
      },
      {
        $group: {
          _id: null,
          totalOutstandingDebt: { $sum: "$outstandingBalance" },
          totalCreditSales: { $sum: "$grandTotal" },
          totalPaidAmount: { $sum: "$totalPaid" },
          unpaidInvoiceCount: { $sum: 1 },
        },
      },
    ]),
  ]);

  // Extract results with defaults
  const [salesData, debtorsData, productsData, expensesData, creditData] = [
    salesResults[0] || {},
    debtorsResults[0] || {},
    productStats[0] || {},
    expensesResults[0] || {},
    creditSalesHistory[0] || {},
  ];

  // Financial calculations
  const cashSales = salesData.totalCashSales || 0;
  const mobileMoneySales = salesData.totalMobileMoneySales || 0;
  const giftSales = salesData.totalGiftSales || 0;
  const creditPaid = debtorsData.totalPaidAmount || 0;

  const totalSales = cashSales + mobileMoneySales + giftSales + creditPaid;
  const totalCost =
    (salesData.totalCost || 0) + (debtorsData.totalCreditCost || 0);
  const grossProfit = totalSales - totalCost;
  const totalExpenses = expensesData.totalExpenses || 0;
  const netProfit = grossProfit - totalExpenses;
  const profitMargin =
    totalSales > 0 ? ((netProfit / totalSales) * 100).toFixed(2) : 0;

  // Response structure
  res.status(200).json({
    status: "success",
    period: {
      year: specifiedYear,
      month: month ? parseInt(month, 10) : undefined,
      startDate,
      endDate,
    },
    financials: {
      totalSales,
      totalCost,
      grossProfit,
      totalExpenses,
      netProfit,
      profitMargin,
    },
    paymentMethodBreakdown: {
      cash: cashSales,
      mobileMoney: mobileMoneySales,
      gift: giftSales,
      creditPaid,
    },
    creditDetails: {
      totalCreditSales: creditData.totalCreditSales || 0,
      totalPaidAmount: creditData.totalPaidAmount || 0,
      outstandingDebt: creditData.totalOutstandingDebt || 0,
      unpaidInvoices: creditData.unpaidInvoiceCount || 0,
    },
    products: {
      count: productsData.totalProductCount || 0,
      quantitySold: salesData.totalQuantitySold || 0,
    },
    expenses: {
      total: totalExpenses,
      count: expensesData.expenseCount || 0,
    },
    transactions: {
      total: salesData.totalSalesCount || 0,
      creditPayments: debtorsData.creditPaymentCount || 0,
    },
  });
});

exports.getSalesAnalyticsByYearMonthOldD = catchAsync(
  async (req, res, next) => {
    const { year, month, period } = req.query;

    const specifiedYear = parseInt(year, 10) || new Date().getFullYear();
    const now = new Date();

    let startDate, endDate;

    // Define start and end dates based on period
    switch (period) {
      case "today":
        startDate = new Date();
        startDate.setHours(0, 0, 0, 0);
        endDate = new Date();
        endDate.setHours(23, 59, 59, 999);
        break;
      case "yesterday":
        startDate = new Date();
        startDate.setDate(startDate.getDate() - 1);
        startDate.setHours(0, 0, 0, 0);
        endDate = new Date(startDate);
        endDate.setHours(23, 59, 59, 999);
        break;
      case "thisweek":
        const dayOfWeek = now.getDay();
        const diffToMonday =
          now.getDate() - dayOfWeek + (dayOfWeek === 0 ? -6 : 1);
        startDate = new Date(now.setDate(diffToMonday));
        startDate.setHours(0, 0, 0, 0);
        endDate = new Date();
        endDate.setHours(23, 59, 59, 999);
        break;
      case "last30days":
        endDate = new Date();
        endDate.setHours(23, 59, 59, 999);
        startDate = new Date();
        startDate.setDate(startDate.getDate() - 29);
        startDate.setHours(0, 0, 0, 0);
        break;
      default:
        if (month) {
          startDate = new Date(
            Date.UTC(specifiedYear, parseInt(month, 10) - 1, 1)
          );
          endDate = new Date(
            Date.UTC(specifiedYear, parseInt(month, 10), 0, 23, 59, 59, 999)
          );
        } else {
          startDate = new Date(Date.UTC(specifiedYear, 0, 1));
          endDate = new Date(Date.UTC(specifiedYear, 11, 31, 23, 59, 59, 999));
        }
    }

    const dateFilter = {
      dateTime: { $gte: startDate, $lte: endDate },
    };

    // Get sales documents directly to avoid double counting due to $unwind
    const salesDocs = await Sales.find({
      ...dateFilter,
      paymentMethod: { $in: ["Cash", "MobileMoney", "Gift"] },
    });

    const debtorsDocs = await DebtorsHistory.find({
      ...dateFilter,
      paymentHistory: { $exists: true, $ne: [] },
    });

    const expensesData = await Expenses.aggregate([
      {
        $match: {
          date: { $gte: startDate, $lte: endDate },
        },
      },
      {
        $group: {
          _id: null,
          totalExpenses: { $sum: "$amount" },
          expenseCount: { $sum: 1 },
        },
      },
    ]);

    let totalCashSales = 0;
    let totalMobileMoneySales = 0;
    let totalGiftSales = 0;
    let totalCost = 0;
    let totalQuantitySold = 0;

    for (const sale of salesDocs) {
      const payment = sale.paymentMethod;
      const grandTotal = sale.grandTotal || 0;

      if (payment === "Cash") totalCashSales += grandTotal;
      else if (payment === "MobileMoney") totalMobileMoneySales += grandTotal;
      else if (payment === "Gift") totalGiftSales += grandTotal;

      for (const product of sale.products) {
        totalCost += (product.count || 0) * (product.costPrice || 0);
        totalQuantitySold += product.count || 0;
      }
    }

    let totalCreditSales = 0;
    let creditPaid = 0;
    let creditCost = 0;
    let outstandingDebt = 0;
    let unpaidInvoices = 0;

    for (const debtor of debtorsDocs) {
      const paidAmount = debtor.paymentHistory?.reduce(
        (sum, entry) => sum + (entry.paid_amount || 0),
        0
      );

      totalCreditSales += debtor.grandTotal || 0;
      creditPaid += paidAmount;
      outstandingDebt += Math.max((debtor.grandTotal || 0) - paidAmount, 0);
      if ((debtor.grandTotal || 0) - paidAmount > 0) unpaidInvoices++;

      for (const product of debtor.products) {
        creditCost += (product.count || 0) * (product.costPrice || 0);
      }
    }

    const totalSales =
      totalCashSales + totalMobileMoneySales + totalGiftSales + creditPaid;
    totalCost += creditCost;
    const grossProfit = totalSales - totalCost;
    const totalExpenses = expensesData[0]?.totalExpenses || 0;
    const netProfit = grossProfit - totalExpenses;
    const profitMargin =
      totalSales > 0 ? ((netProfit / totalSales) * 100).toFixed(2) : "0.00";

    res.status(200).json({
      status: "success",
      period: {
        year: specifiedYear,
        month: month ? parseInt(month, 10) : undefined,
        period: period || null,
        startDate,
        endDate,
      },
      financials: {
        totalSales,
        totalCost,
        grossProfit,
        totalExpenses,
        netProfit,
        profitMargin,
      },
      paymentMethodBreakdown: {
        cash: totalCashSales,
        mobileMoney: totalMobileMoneySales,
        gift: totalGiftSales,
        creditPaid,
      },
      creditDetails: {
        totalCreditSales,
        totalPaidAmount: creditPaid,
        outstandingDebt,
        unpaidInvoices,
      },
      products: {
        quantitySold: totalQuantitySold,
      },
      expenses: {
        total: totalExpenses,
        count: expensesData[0]?.expenseCount || 0,
      },
      transactions: {
        total: salesDocs.length,
        creditPayments: debtorsDocs.reduce(
          (sum, d) => sum + (d.paymentHistory?.length || 0),
          0
        ),
      },
    });
  }
);

exports.getSalesAnalyticsByYearMonth = catchAsync(async (req, res, next) => {
  const { year, month, period } = req.query;
  const specifiedYear = parseInt(year, 10) || new Date().getFullYear();
  const now = new Date();

  let startDate, endDate;

  switch (period) {
    case "today":
      startDate = new Date();
      startDate.setHours(0, 0, 0, 0);
      endDate = new Date();
      endDate.setHours(23, 59, 59, 999);
      break;
    case "yesterday":
      startDate = new Date();
      startDate.setDate(startDate.getDate() - 1);
      startDate.setHours(0, 0, 0, 0);
      endDate = new Date(startDate);
      endDate.setHours(23, 59, 59, 999);
      break;
    case "thisweek":
      const dayOfWeek = now.getDay();
      const diffToMonday =
        now.getDate() - dayOfWeek + (dayOfWeek === 0 ? -6 : 1);
      startDate = new Date(now.setDate(diffToMonday));
      startDate.setHours(0, 0, 0, 0);
      endDate = new Date();
      endDate.setHours(23, 59, 59, 999);
      break;
    case "last30days":
      endDate = new Date();
      endDate.setHours(23, 59, 59, 999);
      startDate = new Date();
      startDate.setDate(startDate.getDate() - 29);
      startDate.setHours(0, 0, 0, 0);
      break;
    default:
      if (month) {
        startDate = new Date(Date.UTC(specifiedYear, parseInt(month) - 1, 1));
        endDate = new Date(
          Date.UTC(specifiedYear, parseInt(month), 0, 23, 59, 59, 999)
        );
      } else {
        startDate = new Date(Date.UTC(specifiedYear, 0, 1));
        endDate = new Date(Date.UTC(specifiedYear, 11, 31, 23, 59, 59, 999));
      }
  }

  // Fetch documents
  const [salesDocs, debtorsDocs, expensesData, debtorPayments] =
    await Promise.all([
      Sales.find({ dateTime: { $gte: startDate, $lte: endDate } }),
      DebtorsHistory.find({ dateTime: { $gte: startDate, $lte: endDate } }),
      Expenses.aggregate([
        {
          $match: { date: { $gte: startDate, $lte: endDate } },
        },
        {
          $group: {
            _id: null,
            total: { $sum: "$amount" },
            count: { $sum: 1 },
          },
        },
      ]),
      DebtorsHistory.aggregate([
        { $unwind: "$paymentHistory" },
        {
          $match: {
            "paymentHistory.date_paid": { $gte: startDate, $lte: endDate },
          },
        },
        {
          $group: {
            _id: null,
            totalPaid: { $sum: "$paymentHistory.paid_amount" },
          },
        },
      ]),
    ]);

  // Initialize accumulators
  let totalSales = 0;
  let totalCost = 0;
  let cashSales = 0;
  let mobileMoneySales = 0;
  let giftSales = 0;
  let totalQuantitySold = 0;
  let totalCreditSales = 0;
  let outstandingDebt = 0;
  let unpaidInvoices = 0;
  let costOfCashSales = 0;
  let costOfDebtorSales = 0;

  // Process direct sales
  for (const sale of salesDocs) {
    totalSales += sale.grandTotal || 0;

    if (sale.paymentMethod === "Cash") cashSales += sale.grandTotal || 0;
    if (sale.paymentMethod === "MobileMoney")
      mobileMoneySales += sale.grandTotal || 0;
    if (sale.paymentMethod === "Gift") giftSales += sale.grandTotal || 0;

    for (const product of sale.products || []) {
      const productCost = (product.costPrice || 0) * (product.count || 0);
      totalCost += productCost;
      totalQuantitySold += product.count || 0;

      if (["Cash", "MobileMoney", "Gift"].includes(sale.paymentMethod)) {
        costOfCashSales += productCost;
      }
    }
  }

  // Process debtor/credit sales
  for (const debtor of debtorsDocs) {
    totalSales += debtor.grandTotal || 0;
    totalCreditSales += debtor.grandTotal || 0;

    const paid = (debtor.paymentHistory || []).reduce(
      (sum, p) => sum + (p.paid_amount || 0),
      0
    );

    const unpaid = Math.max((debtor.grandTotal || 0) - paid, 0);
    outstandingDebt += unpaid;
    if (unpaid > 0) unpaidInvoices++;

    for (const product of debtor.products || []) {
      const productCost = (product.costPrice || 0) * (product.count || 0);
      totalCost += productCost;
      totalQuantitySold += product.count || 0;
      costOfDebtorSales += productCost;
    }
  }

  const amountPaidByDebtors = debtorPayments[0]?.totalPaid || 0;
  const totalExpenses = expensesData[0]?.total || 0;

  const cashReceived =
    cashSales + mobileMoneySales + giftSales + amountPaidByDebtors;
  const grossProfit = totalSales - totalCost;
  const netProfit = grossProfit - totalExpenses;
  const netCashProfit = cashReceived - totalCost - totalExpenses;
  const profitMargin =
    totalSales > 0 ? ((netProfit / totalSales) * 100).toFixed(2) : "0.00";

  res.status(200).json({
    status: "success",
    period: {
      year: specifiedYear,
      month: month ? parseInt(month, 10) : undefined,
      period: period || null,
      startDate,
      endDate,
    },
    financials: {
      totalSales,
      cashReceived,
      totalCost,
      costOfCashSales,
      costOfDebtorSales,
      grossProfit,
      totalExpenses,
      netProfit,
      netCashProfit,
      profitMargin,
    },
    paymentBreakdown: {
      cash: cashSales,
      mobileMoney: mobileMoneySales,
      gift: giftSales,
      credit: amountPaidByDebtors,
    },
    creditSummary: {
      totalCreditSales,
      outstandingDebt,
      unpaidInvoices,
      debtorPayments: amountPaidByDebtors,
    },
    productStats: {
      quantitySold: totalQuantitySold,
    },
    expenses: {
      total: totalExpenses,
      count: expensesData[0]?.count || 0,
    },
    transactions: {
      totalSalesCount: salesDocs.length,
      creditPaymentCount: debtorsDocs.reduce(
        (sum, d) => sum + (d.paymentHistory?.length || 0),
        0
      ),
    },
  });
});

exports.getProductYearlySalesStatistics = catchAsync(async (req, res, next) => {
  const { productId, year } = req.query;

  if (!productId) {
    return res.status(400).json({
      status: "fail",
      message: "Product ID is required in the query.",
    });
  }
  const product = await Product.findById(productId);
  // console.log(product);
  // Default to the current year if no year is specified
  const specifiedYear = parseInt(year, 10) || new Date().getFullYear();

  // Calculate start and end of the year
  const yearStart = new Date(specifiedYear, 0, 1); // January 1st
  const yearEnd = new Date(specifiedYear, 11, 31, 23, 59, 59); // December 31st

  // List of month names
  const monthNames = [
    "January",
    "February",
    "March",
    "April",
    "May",
    "June",
    "July",
    "August",
    "September",
    "October",
    "November",
    "December",
  ];

  // Aggregate sales data for the product within the specified year
  const statistics = await Sales.aggregate([
    {
      $match: {
        dateTime: {
          $gte: yearStart,
          $lte: yearEnd,
        },
        "products._id": productId, // Match sales for the selected product ID
      },
    },
    {
      $unwind: "$products", // Deconstruct the products array
    },
    {
      $match: {
        "products._id": productId, // Filter by the product ID
      },
    },
    {
      $group: {
        _id: {
          month: { $month: "$dateTime" },
        },
        // productName: { $first: "$products.name" },
        totalQuantitySold: { $sum: "$products.count" },
        totalSales: {
          $sum: { $multiply: ["$products.count", "$products.sellingPrice"] },
        },
      },
    },
    {
      $addFields: {
        month: { $arrayElemAt: [monthNames, { $subtract: ["$_id.month", 1] }] }, // Map month number to name
      },
    },
    {
      $sort: { "_id.month": 1 }, // Sort by month
    },
    {
      $project: {
        _id: 0,
        month: 1,
        productName: 1,
        totalQuantitySold: 1,
        totalSales: 1,
      },
    },
  ]);

  // Calculate the grand total sales for the year
  const grandTotalSales = await Sales.aggregate([
    {
      $match: {
        dateTime: {
          $gte: yearStart,
          $lte: yearEnd,
        },
        "products._id": productId, // Match sales for the selected product ID
      },
    },
    {
      $unwind: "$products", // Deconstruct the products array
    },
    {
      $match: {
        "products._id": productId, // Filter by the product ID
      },
    },
    {
      $group: {
        _id: null,
        grandTotal: {
          $sum: { $multiply: ["$products.count", "$products.sellingPrice"] },
        },
      },
    },
    {
      $project: {
        _id: 0,
        grandTotal: 1,
      },
    },
  ]);

  res.status(200).json({
    status: "success",
    productId,
    year: specifiedYear,
    productName: product.name,
    grandTotalSales:
      grandTotalSales.length > 0 ? grandTotalSales[0].grandTotal : 0,
    statistics,
  });
});

exports.getCreditSalesGrandTotalSalesForYear = catchAsync(
  async (req, res, next) => {
    const { year } = req.query;

    // Default to the current year if no year is specified
    const specifiedYear = parseInt(year, 10) || new Date().getFullYear();

    // Fetch total amount expected, total paid amount, and total amount remaining
    const debtorsResults = await DebtorsHistory.aggregate([
      {
        $match: {
          dateTime: {
            $gte: new Date(`${specifiedYear}-01-01T00:00:00.000Z`),
            $lte: new Date(`${specifiedYear}-12-31T23:59:59.999Z`),
          },
        },
      },
      {
        $group: {
          _id: null,
          totalAmountExpected: { $sum: "$grandTotal" }, // Sum of grandTotal for all credit sales
          totalPaidAmount: {
            $sum: {
              $reduce: {
                input: "$paymentHistory",
                initialValue: 0,
                in: { $add: ["$$value", "$$this.paid_amount"] }, // Sum paid_amount in paymentHistory
              },
            },
          },
        },
      },
      {
        $project: {
          totalAmountExpected: 1,
          totalPaidAmount: 1,
          totalAmountRemaining: {
            $subtract: ["$totalAmountExpected", "$totalPaidAmount"],
          }, // Calculate remaining amount
        },
      },
    ]);

    // Fetch total product count from DebtorsHistory
    const productCounts = await DebtorsHistory.aggregate([
      {
        $match: {
          dateTime: {
            $gte: new Date(`${specifiedYear}-01-01T00:00:00.000Z`),
            $lte: new Date(`${specifiedYear}-12-31T23:59:59.999Z`),
          },
        },
      },
      { $unwind: "$products" },
      { $group: { _id: null, totalProductCount: { $sum: "$products.count" } } },
    ]);

    // Extract results from the aggregation
    const totalAmountExpected = debtorsResults[0]?.totalAmountExpected || 0;
    const totalPaidAmount = debtorsResults[0]?.totalPaidAmount || 0;
    const totalAmountRemaining = debtorsResults[0]?.totalAmountRemaining || 0;
    const totalProductCount = productCounts[0]?.totalProductCount || 0;

    res.status(200).json({
      status: "success",
      year: specifiedYear,
      totalAmountExpected,
      totalPaidAmount,
      totalAmountRemaining,
      totalProductCount, // Total count of products sold
    });
  }
);
exports.downloadSales = catchAsync(async (req, res, next) => {
  const date = req.query.date || new Date().toISOString().split("T")[0];

  // Get your sales data (replace with your actual data fetching logic)

  const startOfDay = new Date();
  startOfDay.setHours(0, 0, 0, 0);
  const endOfDay = new Date();
  endOfDay.setHours(23, 59, 59, 999);
  const salesData = await Sales.find({
    // seller: userId,
    createdAt: { $gte: startOfDay, $lte: endOfDay },
  });

  // Generate CSV
  const csvData = generateSalesCSV(salesData);

  // Set headers for file download
  res.setHeader("Content-Type", "text/csv");
  res.setHeader(
    "Content-Disposition",
    `attachment; filename=sales-report-${date}.csv`
  );

  // Send the CSV
  res.status(200).send(csvData);
});

exports.getTopProductsByQuantity = async (req, res) => {
  try {
    const topProducts = await Sales.aggregate([
      // Unwind the products array to work with individual products
      { $unwind: "$products" },

      // Group by product ID and calculate total quantity sold
      {
        $group: {
          _id: "$products._id",
          name: { $first: "$products.name" },
          totalQuantity: { $sum: "$products.quantity" },
          sellingPrice: { $first: "$products.sellingPrice" },
        },
      },

      // Sort by total quantity in descending order
      { $sort: { totalQuantity: -1 } },

      // Limit to top 10 results
      { $limit: 10 },

      // Project to shape the output
      {
        $project: {
          _id: 1,
          name: 1,
          totalQuantity: 1,
          totalAmount: { $multiply: ["$totalQuantity", "$sellingPrice"] },
        },
      },
    ]);

    res.status(200).json({
      success: true,
      data: topProducts,
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      message: "Error fetching top products by quantity",
      error: error.message,
    });
  }
};

exports.getTopProductsByAmount = async (req, res) => {
  try {
    const topProducts = await Sales.aggregate([
      // Unwind the products array
      { $unwind: "$products" },

      // Calculate amount for each product (quantity * sellingPrice)
      {
        $addFields: {
          "products.productAmount": {
            $multiply: ["$products.quantity", "$products.sellingPrice"],
          },
        },
      },

      // Group by product ID and calculate total amount
      {
        $group: {
          _id: "$products._id",
          name: { $first: "$products.name" },
          totalAmount: { $sum: "$products.productAmount" },
          totalQuantity: { $sum: "$products.quantity" },
        },
      },

      // Sort by total amount in descending order
      { $sort: { totalAmount: -1 } },

      // Limit to top 10 results
      { $limit: 10 },
    ]);

    res.status(200).json({
      success: true,
      data: topProducts,
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      message: "Error fetching top products by amount",
      error: error.message,
    });
  }
};

// Combined Controller (Both Reports in One)
exports.getTopProductsOld = async (req, res) => {
  try {
    // Get top by quantity
    const byQuantity = await Sales.aggregate([
      { $unwind: "$products" },
      {
        $group: {
          _id: "$products._id",
          name: { $first: "$products.name" },
          totalQuantity: { $sum: "$products.quantity" },
          sellingPrice: { $first: "$products.sellingPrice" },
        },
      },
      { $sort: { totalQuantity: -1 } },
      { $limit: 10 },
      {
        $project: {
          _id: 1,
          name: 1,
          totalQuantity: 1,
          totalAmount: { $multiply: ["$totalQuantity", "$sellingPrice"] },
        },
      },
    ]);

    // Get top by amount
    const byAmount = await Sales.aggregate([
      { $unwind: "$products" },
      {
        $addFields: {
          "products.productAmount": {
            $multiply: ["$products.quantity", "$products.sellingPrice"],
          },
        },
      },
      {
        $group: {
          _id: "$products._id",
          name: { $first: "$products.name" },
          totalAmount: { $sum: "$products.productAmount" },
          totalQuantity: { $sum: "$products.quantity" },
        },
      },
      { $sort: { totalAmount: -1 } },
      { $limit: 10 },
    ]);

    res.status(200).json({
      success: true,
      data: {
        byQuantity,
        byAmount,
      },
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      message: "Error fetching top products",
      error: error.message,
    });
  }
};
exports.getTopProducts = async (req, res) => {
  try {
    // Get sales data from both collections
    const [salesQuantity, debtorsQuantity] = await Promise.all([
      // Sales collection - by quantity
      Sales.aggregate([
        { $unwind: "$products" },
        {
          $group: {
            _id: "$products._id",
            name: { $first: "$products.name" },
            totalQuantity: { $sum: "$products.quantity" },
            sellingPrice: { $first: "$products.sellingPrice" },
          },
        },
      ]),
      // DebtorsHistory collection - by quantity
      DebtorsHistory.aggregate([
        { $unwind: "$products" },
        {
          $group: {
            _id: "$products._id",
            name: { $first: "$products.name" },
            totalQuantity: { $sum: "$products.quantity" },
            sellingPrice: { $first: "$products.sellingPrice" },
          },
        },
      ]),
    ]);

    const [salesAmount, debtorsAmount] = await Promise.all([
      // Sales collection - by amount
      Sales.aggregate([
        { $unwind: "$products" },
        {
          $addFields: {
            "products.productAmount": {
              $multiply: ["$products.quantity", "$products.sellingPrice"],
            },
          },
        },
        {
          $group: {
            _id: "$products._id",
            name: { $first: "$products.name" },
            totalAmount: { $sum: "$products.productAmount" },
            totalQuantity: { $sum: "$products.quantity" },
          },
        },
      ]),
      // DebtorsHistory collection - by amount
      DebtorsHistory.aggregate([
        { $unwind: "$products" },
        {
          $addFields: {
            "products.productAmount": {
              $multiply: ["$products.quantity", "$products.sellingPrice"],
            },
          },
        },
        {
          $group: {
            _id: "$products._id",
            name: { $first: "$products.name" },
            totalAmount: { $sum: "$products.productAmount" },
            totalQuantity: { $sum: "$products.quantity" },
          },
        },
      ]),
    ]);

    // Combine results from both collections
    const combinedQuantity = [...salesQuantity, ...debtorsQuantity].reduce(
      (acc, curr) => {
        const existing = acc.find(
          (item) => item._id.toString() === curr._id.toString()
        );
        if (existing) {
          existing.totalQuantity += curr.totalQuantity;
          existing.totalAmount = existing.totalQuantity * existing.sellingPrice;
        } else {
          acc.push({
            ...curr,
            totalAmount: curr.totalQuantity * curr.sellingPrice,
          });
        }
        return acc;
      },
      []
    );

    const combinedAmount = [...salesAmount, ...debtorsAmount].reduce(
      (acc, curr) => {
        const existing = acc.find(
          (item) => item._id.toString() === curr._id.toString()
        );
        if (existing) {
          existing.totalAmount += curr.totalAmount;
          existing.totalQuantity += curr.totalQuantity;
        } else {
          acc.push(curr);
        }
        return acc;
      },
      []
    );

    // Sort and limit results
    const byQuantity = combinedQuantity
      .sort((a, b) => b.totalQuantity - a.totalQuantity)
      .slice(0, 10)
      .map((item) => ({
        _id: item._id,
        name: item.name,
        totalQuantity: item.totalQuantity,
        totalAmount: item.totalAmount,
      }));

    const byAmount = combinedAmount
      .sort((a, b) => b.totalAmount - a.totalAmount)
      .slice(0, 10);

    res.status(200).json({
      success: true,
      data: {
        byQuantity,
        byAmount,
      },
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      message: "Error fetching top products",
      error: error.message,
    });
  }
};
