const Product = require("../models/productModel");
const Activity = require("../models/activityTrackerModel");
const StokeTake = require("../models/stokeTakeModel");
const TallyCard = require("../models/tallyCardModel");
const catchAsync = require("../utils/catchAsync");
const AppError = require("../utils/appError");

exports.createStokeTake = catchAsync(async (req, res, next) => {
  const { cart, grandQuantity } = req.body;

  if (!cart) {
    return next(new AppError("Something is wrong", 400));
  }

  try {
    // Check if any cart item has null or empty count
    for (const item of cart) {
      if (item.count === null || item.count === "") {
        return next(new AppError(`${item.name} has an empty count`, 400));
      }
    }

    // If all cart items have valid counts, proceed with creating StokeTake and updating products
    for (const item of cart) {
      const product = await Product.findById(item._id);
      if (!product) {
        throw new Error(`Product with ID ${item._id} not found`);
      }
      const count = item.count || 0;
      await Promise.all([
        countAndUpdateProduct(item._id, count),
        createTallyCardEntry(
          item._id,
          item.count,
          product.quantity,
          item.stokeId,
          req.user._id
        ),
      ]);
    }

    const newStokeTake = new StokeTake({
      user: req.user._id,
      grandQuantity,
      products: cart,
    });

    await newStokeTake.save();

    // Log stock take activity without cart details
    const activity = new Activity({
      user: req.user._id,
      activityType: "stock_take",
      details: {
        stock: newStokeTake,
      },
    });
    await activity.save();

    res.status(200).json({
      msg: "StokeTake success!",
      newStokeTake,
    });
  } catch (error) {
    return next(error);
  }
});

const countAndUpdateProduct = async (id, count) => {
  try {
    // Check if the document exists before attempting to update
    const existingProduct = await Product.findById(id);

    if (!existingProduct) {
      console.error(`Product with id ${id} not found`);
      return null; // Return null if the document doesn't exist
    }

    // Update the document
    const updatedProduct = await Product.findOneAndUpdate(
      { _id: id },
      { quantity: Number(count) },
      { new: true }
    );

    return updatedProduct;
  } catch (error) {
    console.error("Error updating product:", error);
    throw error; // Re-throw the error to propagate it
  }
};

const createTallyCardEntry = async (
  productId,
  quantityCount,
  productQuantity,
  stokeId,
  userId
) => {
  try {
    const newTallyEntry = new TallyCard({
      product: productId,
      inflow:
        Number(quantityCount) >= Number(productQuantity)
          ? Number(quantityCount) - Number(productQuantity)
          : 0,
      // quantityCount > productQuantity ? quantityCount - productQuantity : 0,
      outflow:
        Number(quantityCount) >= Number(productQuantity)
          ? 0
          : Number(productQuantity) - Number(quantityCount),
      previousQty: Number(productQuantity),
      stockBalance: Number(quantityCount),
      description: "StockTake",
      user: userId,
      date: new Date(),
      stokeId,
    });
    await newTallyEntry.save();
  } catch (error) {
    console.error("Error creating tally card entry:", error);
    throw error;
  }
};

exports.getAllStokeTake = catchAsync(async (req, res, next) => {
  const stokes = await StokeTake.find({}).sort({ stockTakeTime: -1 });
  res.status(200).send(stokes);
});

exports.getStokeTakeByDate = catchAsync(async (req, res, next) => {
  if (req.query.startdate && req.query.enddate) {
    startDate = new Date(req.query.startdate);
    startDate.setHours(0, 0, 0, 0);

    endDate = new Date(req.query.enddate);
    endDate.setHours(23, 59, 59, 999);
  } else {
    // beginning of current day
    var startDate = new Date();
    startDate.setHours(0, 0, 0, 0);

    // end of current day
    var endDate = new Date();
    endDate.setHours(23, 59, 59, 999);
  }
  const docs = await StokeTake.find({
    stockTakeTime: { $gte: startDate.toJSON(), $lte: endDate.toJSON() },
  })
    .sort({ stockTakeTime: -1 })
    .populate("user", "name");
  var result = {
    stockTakeTime: startDate,
  };

  if (docs) {
    var grandQuantity = docs.reduce(function (p, c) {
      return p + c.grandQuantity;
    }, 0);
    result.grandQuantity = grandQuantity;
    res.status(200).send({
      result,
      docs,
    });
  }
});

exports.getStokeTakeReport = catchAsync(async (req, res, next) => {
  const { selectedDate } = req.query;
  if (selectedDate) {
    startDate = new Date(selectedDate);
    startDate.setHours(0, 0, 0, 0);

    endDate = new Date(selectedDate);
    endDate.setHours(23, 59, 59, 999);
  } else {
    // beginning of current day
    var startDate = new Date();
    startDate.setHours(0, 0, 0, 0);

    // end of current day
    var endDate = new Date();
    endDate.setHours(23, 59, 59, 999);
  }
  const docs = await StokeTake.find({
    stockTakeTime: { $gte: startDate.toJSON(), $lte: endDate.toJSON() },
  })
    .sort({ stockTakeTime: -1 })
    .populate("user", "name");
  var result = {
    stockTakeTime: startDate,
  };
  if (docs) {
    var grandQuantity = docs.reduce(function (p, c) {
      return p + Number(c.grandQuantity);
    }, 0);
    result.grandQuantity = Number(grandQuantity);
    res.status(200).send({
      result,
      docs,
    });
  }
});

exports.getAllUniqueStockTakeTimes = async (req, res) => {
  try {
    // Use aggregation pipeline to get distinct and sorted stockTakeTime values
    const uniqueStockTakeTimes = await StokeTake.aggregate([
      // Group by stockTakeTime to get distinct values
      { $group: { _id: "$stockTakeTime" } },
      // Sort the distinct values by stockTakeTime (optional)
      { $sort: { _id: 1 } },
      // Project to reshape the output and include only the _id field (stockTakeTime)
      { $project: { _id: 0, stockTakeTime: "$_id" } },
    ]);

    // Extract stockTakeTime values from the result
    const stockTakeTimes = uniqueStockTakeTimes.map(
      (entry) => entry.stockTakeTime
    );
    // console.log(stockTakeTimes);
    res.status(200).json(stockTakeTimes);
  } catch (error) {
    res.status(500).json({ error: "Internal Server Error" });
  }
};

// delete
exports.deleteStokeTake = catchAsync(async (req, res, next) => {
  const { stokeId } = req.query;
  const stoke = await StokeTake.findById(stokeId);
  if (!stoke) {
    return next(new AppError("StokeTake not found", 404));
  }
  const activity = new Activity({
    user: req.user._id,
    activityType: "delete_stocktake",
    details: {
      deletedProduct,
    },
  });
  await activity.save();
  const data = await StokeTake.findByIdAndRemove(stoke._id);
  res.status(200).send({ status: "Success" });
});

// Add this function to calculate the sum of count exports
const calculateTotalCountExports = async () => {
  try {
    const allProducts = await StokeTake.find();
    let totalCountExports = 0;
    allProducts.forEach((product) => {
      product.products.forEach((item) => {
        totalCountExports += Number(item.count);
      });
    });
    return totalCountExports;
  } catch (error) {
    console.error("Error calculating total count exports:", error);
    throw error;
  }
};

// Update your updateStokeTake function to include calculating total count exports
exports.updateStokeTake = catchAsync(async (req, res, next) => {
  let { receiptId, stokeId, newQuantity, itemDiscrepancyReasons } = req.body;

  const { productId } = req.params;
  const product = await Product.findOne({ _id: productId });
  const newUpdate = await Product.findByIdAndUpdate(productId, {
    quantity: Number(newQuantity),
  });

  const stoketake = await StokeTake.findOne({ _id: receiptId });

  const stoketakepro = stoketake.products.find(
    (product) => product.stokeId === stokeId
  );
  const varianceQty = Number(newQuantity) - Number(stoketakepro.quantity);
  const varianceAmt = Number(varianceQty) * Number(stoketakepro.costPrice);
  // console.log(newUpdate.quantity);
  createTallyCardUpdate(
    productId,
    newQuantity,
    newUpdate.quantity,
    product.quantity,
    req.user._id
  );
  const updatedProduct = await StokeTake.updateOne(
    { "products.stokeId": stokeId },
    {
      $set: {
        "products.$.count": Number(newQuantity),
        "products.$.resolutionQty": Number(newQuantity),
        "products.$.varianceQty": Number(varianceQty),
        "products.$.varianceAmt": Number(varianceAmt),
        "products.$.itemDiscrepancyReasons": itemDiscrepancyReasons,
      },
    },
    {
      new: true,
    }
  );

  const activity = new Activity({
    user: req.user._id,
    activityType: "update_stocktake",
    details: {
      receiptId,
      stokeId,
      newQuantity,
      itemDiscrepancyReasons,
    },
  });

  await activity.save();

  // Save the updated document to the database
  // await stokeEntry.save();
  // Calculate total count exports after updating the stock
  const totalCountExports = await calculateTotalCountExports();
  await StokeTake.findOneAndUpdate(
    { _id: receiptId },
    {
      grandQuantity: totalCountExports,
    },
    {
      new: true,
    }
  );

  res.status(200).json(updatedProduct);
});

const createTallyCardUpdate = async (
  productId,
  newQuantity,
  quantity,
  prevQty,
  userId
) => {
  try {
    // console.log(quantityCount);
    const newTallyEntry = new TallyCard({
      product: productId,
      inflow:
        Number(newQuantity) >= Number(quantity)
          ? Number(newQuantity) - Number(quantity)
          : 0,
      outflow:
        Number(newQuantity) > Number(quantity)
          ? 0
          : Number(quantity) - Number(newQuantity),
      previousQty: Number(prevQty),
      stockBalance: Number(newQuantity),
      description: "Stock Take Update",
      user: userId,
      date: new Date(),
    });
    await newTallyEntry.save();
  } catch (error) {
    console.error("Error creating tally card entry:", error);
    throw error;
  }
};

exports.deleteProductInStoke = catchAsync(async (req, res) => {
  const { stokeId, productId } = req.params;

  try {
    // Find the sales entry with the specified salesId
    const stokesEntry = await StokeTake.findById(stokeId);

    // Check if the sales entry exists
    if (!stokesEntry) {
      return res.status(404).json({ error: "StokeTake entry not found" });
    }
    // products in a stoke
    const products = stokesEntry.products.length;

    // Find the product being deleted from the sales
    const deletedProduct = stokesEntry.products.find(
      (product) => String(product._id) === productId
    );

    // Check if the product exists
    if (!deletedProduct) {
      return res
        .status(404)
        .json({ error: "Product not found in the sales entry" });
    }
    const activity = new Activity({
      user: req.user._id,
      activityType: "delete_product_instock",
      details: {
        deletedProduct,
      },
    });
    await activity.save();
    // Update the product count in the Product schema
    // await Product.findByIdAndUpdate(deletedProduct._id, {
    //   $inc: { quantity: deletedProduct.count },
    // });

    // Pull the product from the products array
    const stokeEntry = await StokeTake.findByIdAndUpdate(
      stokeId,
      {
        $pull: { products: { _id: productId } },
      },
      { new: true }
    );

    // Update the sales entry with the new grandQuantity and quantitySold
    // stokeEntry.grandQuantity = grandQuantity;

    await stokeEntry.save();

    if (products === 1) {
      await StokeTake.findByIdAndDelete(stokeId);
      return res.json({ message: "StokeTake entry deleted successfully" });
    }
    // If both grandTotal and quantitySold are zero, delete the sales entry

    // Send a success response
    res
      .status(200)
      .json({ message: "Product deleted successfully", stokeEntry });
  } catch (error) {
    console.error("Error deleting product:", error);
    res.status(500).json({ error: "Internal server error" });
  }
});

exports.updateStokeTakeProductDateTime = catchAsync(async (req, res, next) => {
  let { stockId, stockTakeTime } = req.body;
  const tallyProduct = await TallyCard.findOne({ stokeId: stockId });
  // console.log(tallyProduct);
  if (!tallyProduct) {
    throw new Error(`Product with not found`);
  }
  const updatedProduct = await TallyCard.findOneAndUpdate(
    { stokeId: stockId },
    {
      date: stockTakeTime,
    },
    {
      new: true,
    }
  );

  res.status(200).json(updatedProduct);
});

exports.updateInvoiceStokeTakeDate = catchAsync(async (req, res, next) => {
  const { stokeDate } = req.body;
  const { id } = req.params;
  // return console.log(stokeDate);
  // Find the StokeTake record by ID
  const stoketake = await StokeTake.findById(id);

  if (!stoketake) {
    return res.status(404).json({ error: "StokeTake record not found" });
  }

  await StokeTake.findByIdAndUpdate(
    id,
    {
      stockTakeTime: stokeDate,
    },
    {
      new: true,
    }
  );
  // Iterate over each product in the purchase record
  for (const product of stoketake.products) {
    // Assuming each product has a stokeId
    const productStokeId = product.stokeId;

    // Find the product in the TallyCard schema by its stokeId
    const tallyCardProduct = await TallyCard.findOne({
      stokeId: productStokeId,
    });

    if (tallyCardProduct) {
      // Update the date and time of the product in the TallyCard schema
      tallyCardProduct.date = stokeDate;
      await tallyCardProduct.save();
    } else {
      console.log(
        `Product with stokeId ${productStokeId} not found in TallyCard.`
      );
      // Handle the case where the product is not found in the TallyCard schema
    }
  }

  res.status(200).json({ message: "StokeId date updated successfully" });
});
