const Suppliers = require("../models/suppliersModel");
const StartingBalance = require("../models/startingBalanceModel");
const Activity = require("../models/activityTrackerModel");
const catchAsync = require("../utils/catchAsync");
const AppError = require("../utils/appError");
const slugify = require("slugify");
const { generate10DigitUUID } = require("../helpers/generater");
const Purchase = require("../models/purchaseModel");
const StartingBalanceHistory = require("../models/startingBalanceHistoryModel");

// create contact Suppliers
exports.createSuppliers = catchAsync(async (req, res) => {
  const { name, contact, address } = req.body;
  let slug = slugify(name) + generate10DigitUUID();

  const suppliers = await Suppliers({
    slug,
    name,
    contact,
    address,
    user: req.user._id,
  }).save();
  res.status(200).send(suppliers);
});

// get all contact Suppliers
exports.getAllSuppliers = catchAsync(async (req, res) => {
  const suppliers = await Suppliers.find({})
    .sort({ createdAt: -1 })
    .populate("user", "name");

  res.status(200).send(suppliers);
});

// get single contact Suppliers
exports.getSingleSuppliers = catchAsync(async (req, res, next) => {
  const { slug } = req.params;
  const suppliers = await Suppliers.findOne({ slug }).populate(
    "user",
    "name contactNum"
  );
  if (!suppliers) {
    return next(new AppError("Suppliers not found", 404));
  }
  res.status(200).send(suppliers);
});

// update expeses
exports.updateSuppliers = catchAsync(async (req, res, next) => {
  const { name, contact, address } = req.body;
  const { slug } = req.params;
  const suppliersData = await Suppliers.findOne({ slug });
  const updatedSuppliers = await Suppliers.findOneAndUpdate(
    { slug: suppliersData.slug },
    {
      slug:
        name === suppliersData?.name
          ? suppliersData?.name
          : slugify(name) + generate10DigitUUID(),
      name: name,
      contact: contact,
      address: address,
    },
    {
      new: true,
    }
  );
  res.status(200).send(updatedSuppliers);
});

// delete Suppliers
exports.deleteSuppliers = catchAsync(async (req, res, next) => {
  const suppliers = await Suppliers.findById(req.params.id);
  if (!suppliers) {
    return next(new AppError("Expense not found", 404));
  }

  await Suppliers.findByIdAndDelete({ _id: suppliers._id });
  res.status(200).send({ status: "Success" });
});

exports.supplierProducts = catchAsync(async (req, res, next) => {
  const { period, year, month } = req.query;
  const { id } = req.params;
  let startDate, endDate;
  const now = new Date();

  if (period === "weekly") {
    // Set start and end of the current week
    startDate = new Date(now);
    startDate.setDate(now.getDate() - now.getDay()); // Start of the week
    startDate.setHours(0, 0, 0, 0);

    endDate = new Date(startDate);
    endDate.setDate(startDate.getDate() + 6); // End of the week
    endDate.setHours(23, 59, 59, 999);
  } else if (period === "monthly" || (year && month)) {
    // If period is "monthly" or year and month are specified, set start and end of the specified month
    const specifiedYear = parseInt(year) || now.getFullYear();
    const specifiedMonth = month ? parseInt(month) - 1 : now.getMonth(); // Month is zero-indexed

    startDate = new Date(specifiedYear, specifiedMonth, 1); // First day of the month
    startDate.setHours(0, 0, 0, 0);

    endDate = new Date(specifiedYear, specifiedMonth + 1, 0); // Last day of the month
    endDate.setHours(23, 59, 59, 999);
  } else if (period === "yearly") {
    // Set start and end of the current year
    startDate = new Date(now.getFullYear(), 0, 1); // First day of the year
    startDate.setHours(0, 0, 0, 0);

    endDate = new Date(now.getFullYear(), 11, 31); // Last day of the year
    endDate.setHours(23, 59, 59, 999);
  } else if (year) {
    // If only year is provided, set start and end of that year
    startDate = new Date(parseInt(year), 0, 1); // First day of the specified year
    startDate.setHours(0, 0, 0, 0);

    endDate = new Date(parseInt(year), 11, 31); // Last day of the specified year
    endDate.setHours(23, 59, 59, 999);
  }

  // Construct the query object
  const query = { supplier: id };

  // If a date range was specified, add the date range to the query
  if (startDate && endDate) {
    query.createdAt = {
      $gte: startDate.toISOString(),
      $lte: endDate.toISOString(),
    };
  }

  const suppliersData = await Purchase.find(query).populate(
    "supplier",
    "_id name"
  );

  // Calculate total sales amount
  const grandTotalAmount = suppliersData.reduce((p, c) => p + c.amount, 0.0);

  res.status(200).send({ data: suppliersData, amount: grandTotalAmount });
});

// Add a starting balance for a supplier
// "/starting-balance",

exports.createOrUpdateStartingBalance = catchAsync(async (req, res) => {
  try {
    const { supplierId, amount } = req.body;

    // Check if a starting balance already exists for this supplier
    const existingBalance = await StartingBalance.findOne({
      supplier: supplierId,
    });

    if (existingBalance) {
      // Update the existing starting balance
      existingBalance.amount = amount;
      await existingBalance.save();
      return res.status(200).json(existingBalance);
    }

    // Create a new starting balance record if it doesn't exist
    const startingBalance = new StartingBalance({
      supplier: supplierId,
      amount,
    });

    await startingBalance.save();
    res.status(201).json(startingBalance);
  } catch (error) {
    res
      .status(400)
      .json({ message: "Error creating or updating starting balance", error });
  }
});

exports.updateStartingBalance = catchAsync(async (req, res) => {
  try {
    const { supplierId } = req.params;
    const { amount } = req.body;

    // Find the existing starting balance for this supplier
    const startingBalance = await StartingBalance.findOne({
      supplier: supplierId,
    });

    if (!startingBalance) {
      return res
        .status(404)
        .json({ message: "Starting balance not found for this supplier." });
    }

    // Update balance and notes
    startingBalance.amount = amount;
    await startingBalance.save();

    res.status(200).json(startingBalance);
  } catch (error) {
    res.status(500).json({ message: "Error updating starting balance", error });
  }
});

// Get all starting balances for a supplier
// "suppliers/:id/starting-balances"
exports.getStartingBalance = catchAsync(async (req, res) => {
  try {
    const startingBalances = await StartingBalance.findOne({
      supplier: req.params.id,
    });
    res.send(startingBalances);
  } catch (error) {
    res.status(500).send(error);
  }
});

// Repay Starting Balance
exports.repayStartingBalance = catchAsync(async (req, res) => {
  const { startingBalanceId, repaidAmount, notes } = req.body;
  // Find the starting balance
  const startingBalance = await StartingBalance.findById(startingBalanceId);
  if (!startingBalance) {
    return res.status(404).json({ message: "Starting balance not found" });
  }

  // Check if the repaid amount is valid
  if (repaidAmount <= 0 || repaidAmount > startingBalance.amount) {
    return res.status(400).json({ message: "Invalid repayment amount" });
  }

  // Update the starting balance
  const oldAmount = startingBalance.amount;
  startingBalance.amount -= repaidAmount;
  await startingBalance.save();

  // Create a StartingBalanceHistory entry
  const historyEntry = await StartingBalanceHistory.create({
    startingBalance: startingBalanceId,
    oldAmount,
    newAmount: startingBalance.amount,
    repaidAmount,
    notes,
    updatedBy: req.user._id,
  });

  res.status(200).json({
    message: "Starting balance repaid successfully",
    startingBalance,
    historyEntry,
  });
});

// Update StartingBalanceHistory
exports.updateStartingBalanceHistory = catchAsync(async (req, res) => {
  const { historyId } = req.params;
  const { notes } = req.body;

  // Find the history entry
  const historyEntry = await StartingBalanceHistory.findById(historyId);
  if (!historyEntry) {
    return res.status(404).json({ message: "History entry not found" });
  }

  // Update the notes
  historyEntry.notes = notes;
  await historyEntry.save();

  res.status(200).json({
    message: "StartingBalanceHistory updated successfully",
    historyEntry,
  });
});

// Get all StartingBalanceHistory entries for a specific StartingBalance
exports.getStartingBalanceHistories = catchAsync(async (req, res) => {
  const { startingBalanceId } = req.params;

  // Find all history entries for the given StartingBalance
  const histories = await StartingBalanceHistory.find({
    startingBalance: startingBalanceId,
  })
    .sort({ createdAt: -1 })
    .populate("startingBalance", "amount") // Optional: Populate startingBalance details
    .populate("updatedBy", "name email"); // Optional: Populate updatedBy user details

  if (!histories || histories.length === 0) {
    return res.status(404).json({
      message: "No history entries found for this starting balance",
    });
  }

  res.status(200).json(histories);
});

exports.getGrandTotalOwnBySupplier = async (req, res, next) => {
  try {
    const { supplierId } = req.params;

    if (!supplierId) {
      return res.status(400).json({
        success: false,
        message: "Supplier ID is required",
      });
    }

    const supplier = await Suppliers.findById(supplierId);
    if (!supplier) {
      return res.status(404).json({
        success: false,
        message: "Supplier not found",
      });
    }
    const startingBalances = await StartingBalance.findOne({
      supplier: supplierId,
    });
    // Find all purchases for the supplier
    const purchases = await Purchase.find({ supplier: supplierId }).sort({
      createdAt: -1,
    });

    if (!purchases?.length) {
      return res.status(200).json({
        success: true,
        SupplierName: supplier.name,
        grandRemainingBalance: 0,
        data: [],
        message: "No purchases found for this supplier",
      });
    }

    // Format response to match the required structure
    const formattedPurchases = await Promise.all(
      purchases.map(async (purchase) => {
        return {
          purchaseId: purchase._id,
          remainingBalance: purchase.balance || 0,
          createdAt: purchase.createdAt,
        };
      })
    );

    // Filter purchases to include only those with payments or unpaid balances
    const filteredPurchases = formattedPurchases.filter(
      (purchase) =>
        purchase.payments?.length > 0 || purchase.remainingBalance > 0
    );

    const grandRemainingBalance = filteredPurchases.reduce(
      (sum, purchase) => sum + purchase.remainingBalance,
      0
    );

    res.status(200).json({
      grandTotalBalanceAmountOwn:
        grandRemainingBalance + startingBalances.amount,
    });
  } catch (error) {
    next(error);
  }
};
