const TallyCard = require("../models/tallyCardModel");
const Activity = require("../models/activityTrackerModel");

// Controller function to update a TallyCard document
exports.updateTallyCard = async (req, res) => {
  const { id } = req.params; // TallyCard ID from the URL
  const { description, inflow, outflow, previousQty, stockBalance, date } =
    req.body;

  try {
    // Find the TallyCard document by ID and update the specified fields
    const updatedTallyCard = await TallyCard.findByIdAndUpdate(
      id,
      {
        description,
        inflow,
        outflow,
        previousQty,
        stockBalance,
        date,
      },
      { new: true, runValidators: true } // Return the updated document and validate input
    );

    // If no TallyCard document was found, return a 404 error
    if (!updatedTallyCard) {
      return res.status(404).json({ message: "TallyCard not found" });
    }
    const activity = new Activity({
      user: req.user._id,
      activityType: "tallycard_updated",
      details: updatedTallyCard,
    });
    // console.log(activity);
    await activity.save();
    // Return the updated TallyCard document
    res.status(200).json(updatedTallyCard);
  } catch (error) {
    // Handle errors and respond with a 500 status
    res
      .status(500)
      .json({ message: "Error updating TallyCard", error: error.message });
  }
};
exports.deleteTallyCard = async (req, res) => {
  const { id } = req.params; // TallyCard ID from the URL
  try {
    // Find the TallyCard document by ID and update the specified fields
    const tallyCard = await TallyCard.findByIdAndDelete(id);
    // If no TallyCard document was found, return a 404 error
    if (!tallyCard) {
      return res.status(404).json({ message: "TallyCard not found" });
    }
    // Return the updated TallyCard document
    res.status(200).json(tallyCard);
  } catch (error) {
    // Handle errors and respond with a 500 status
    res
      .status(500)
      .json({ message: "Error updating TallyCard", error: error.message });
  }
};

exports.getDataByDateRange = async (req, res) => {
  const { productId } = req.params;
  const { user } = req.query;
  if (req.query.startDate && req.query.endDate) {
    startDate = new Date(req.query.startDate);
    startDate.setHours(0, 0, 0, 0);
    endDate = new Date(req.query.endDate);
    endDate.setHours(23, 59, 59, 999);
  } else {
    // beginning of current day
    var startDate = new Date();
    startDate.setHours(0, 0, 0, 0);

    // end of current day
    var endDate = new Date();
    endDate.setHours(23, 59, 59, 999);
  }
  try {
    let query = { product: productId };
    if (user) {
      query.user = user;
    }

    if (startDate) {
      query.date = { $gte: startDate.toJSON(), $lte: endDate.toJSON() };
    }
    const result = await TallyCard.find(query)
      .populate("user", "name")
      .populate("product", "name initialQty quantity")
      .sort({ date: 1 });

    // Calculate totalInflow
    let totalInflow = 0;
    result.forEach((item) => {
      totalInflow += item.inflow;
    });

    // Calculate totalOutflow
    let totalOutflow = 0;
    result.forEach((item) => {
      totalOutflow += item.outflow;
    });

    // Determine stockBalance as the stock balance of the last document (if any)
    let stockBalance = 0;

    result.forEach((item) => {
      stockBalance += item.stockBalance;
    });

    // Return the result along with separately calculated totalInflow, totalOutflow, and stockBalance
    res.status(200).json({
      totalInflow,
      totalOutflow,
      stockBalance,
      data: result,
    });
  } catch (error) {
    console.error("Error:", error);
    res.status(500).json({ message: error.message || "Internal server error" });
  }
};
