const Transfer = require("../models/transferModel");
const TallyCard = require("../models/tallyCardModel");
const Activity = require("../models/activityTrackerModel");
const Shop = require("../models/shopsModel");
const Product = require("../models/productModel");
const catchAsync = require("../utils/catchAsync");
const AppError = require("../utils/appError");

exports.createTransfer = catchAsync(async (req, res, next) => {
  const {
    transferType,
    cart,
    amount,
    selectedShop,
    grandQuantity,
    purchaseTime,
    documentNumber,
  } = req.body;

  const currentTime = purchaseTime || new Date();

  // Create the transfer document
  const transfer = await Transfer.create({
    user: req.user._id,
    shop: selectedShop,
    quantity: grandQuantity,
    products: cart,
    purchaseTime: currentTime,
    amount,
    documentNumber,
    transferType,
  });

  // Prepare bulk update operations and tally card entries
  const bulkOps = cart.map(
    ({ _id: productId, quantity, count, transferId }) => {
      const adjustedQuantity =
        transferType === "transferIn"
          ? Number(quantity) + Number(count)
          : Number(quantity) - Number(count);

      const createTallyCardEntry =
        transferType === "transferIn"
          ? createTallyCardEntryTransferIn
          : createTallyCardEntryTransferOut;

      createTallyCardEntry(
        productId,
        adjustedQuantity,
        count,
        req.user._id,
        currentTime,
        transferId
      );

      return {
        updateOne: {
          filter: { _id: productId },
          update: { $set: { quantity: adjustedQuantity } },
        },
      };
    }
  );

  // Execute bulk update operations
  await Product.bulkWrite(bulkOps);

  // Log transfer activity
  await Activity.create({
    user: req.user._id,
    activityType: "transfer",
    details: {
      cart,
      amount,
      selectedShop,
      grandQuantity,
      purchaseTime: currentTime,
      transferType,
    },
  });

  // Respond with the created transfer
  res.status(200).json(transfer);
});
const createTallyCardEntryTransferIn = async (
  productId,
  grandQuantity,
  quantity,
  userId,
  purchaseTime,
  transferId
) => {
  try {
    const newTallyEntry = new TallyCard({
      product: productId,
      inflow: Number(quantity),
      outflow: 0,
      previousQty: Number(grandQuantity) - Number(quantity),
      stockBalance: Number(grandQuantity),
      description: "Transfer In",
      user: userId,
      date: purchaseTime ? purchaseTime : new Date(),
      transferId,
    });
    await newTallyEntry.save();
  } catch (error) {
    console.error("Error creating tally card entry:", error);
    throw error;
  }
};
const createTallyCardEntryTransferOut = async (
  productId,
  grandQuantity,
  quantity,
  userId,
  purchaseTime,
  transferId
) => {
  try {
    const newTallyEntry = new TallyCard({
      product: productId,
      inflow: 0,
      outflow: Number(quantity),
      previousQty: Number(grandQuantity) + Number(quantity),
      stockBalance: grandQuantity,
      description: "Transfer Out",
      user: userId,
      date: purchaseTime ? purchaseTime : new Date(),
      transferId,
    });
    await newTallyEntry.save();
  } catch (error) {
    console.error("Error creating tally card entry:", error);
    throw error;
  }
};

exports.getTransfer = catchAsync(async (req, res, next) => {
  const shop = await Shop.findOne({ slug: req.params.shopId });
  const transfer = await Transfer.find({ shop: shop._id })
    .populate("shop", "slug shopname")
    .populate("user", "name email contactNum")
    .sort({
      purchaseTime: -1,
    });
  res.status(200).send(transfer);
});

exports.removeProductInTrans = async (req, res, next) => {
  const { productId, transferId } = req.params;

  const transfer = await Transfer.findById({ _id: transferId }).exec();
  if (!transfer) {
    return next(new AppError("Transfer not found", 404));
  }
  const product = transfer.products.find(
    (product) => product._id.toString() === productId
  );
  if (!product) {
    return next(new AppError("product not found", 404));
  }
  if (transfer?.transferType === "transferIn") {
    const newUpdate = await Product.findByIdAndUpdate(product._id, {
      $inc: { quantity: -product.count },
    });
    createTallyCardEntryTransferInDelete(
      product._id,
      newUpdate.quantity,
      product.count,
      req.user._id
    );
  } else if (transfer?.transferType === "transferOut") {
    const newUpdate = await Product.findByIdAndUpdate(product._id, {
      $inc: { quantity: Number(product.count) },
    });
    createTallyCardEntryTransferOutDelete(
      product._id,
      newUpdate.quantity,
      product.count,
      req.user._id
    );
  } else {
    throw new Error("Invalid transferType");
  }
  // Update the product count in the Product schema
  // createTallyCardEntryTransferDelete(
  //   product._id,
  //   newUpdate.quantity,
  //   product.count,
  //   req.user._id
  // );
  const newQty = Number(transfer.quantity) - Number(product.count);
  const newAmount = Number(transfer.amount) - Number(product.sellingPrice);
  // return console.log(newAmount);
  await Transfer.findByIdAndUpdate(
    transfer._id,
    {
      quantity: Number(newQty),
      amount: Number(newAmount),
    },
    {
      new: true,
    }
  );
  await Transfer.findByIdAndUpdate(transfer._id, {
    $pull: { products: { _id: productId } },
  }).exec();

  if (transfer?.products?.length - 1 === 0) {
    await Transfer.findByIdAndDelete(transferId);
    return res.json({ message: "transfer entry deleted successfully" });
  }
  const activity = new Activity({
    user: req.user._id,
    activityType: "transfer_product_delete",
    details: {
      quantity: newQty,
      amount: newAmount,
      transferId: transferId,
      productId: productId,
    },
  });
  await activity.save();
  res.status(200).json({ flag: true, message: "success" });
};

const createTallyCardEntryTransferInDelete = async (
  productId,
  productQuantity,
  quantity,
  userId
) => {
  try {
    const newTallyEntry = new TallyCard({
      product: productId,
      inflow: Number(productQuantity) - Number(quantity),
      outflow: quantity,
      previousQty: Number(productQuantity),
      stockBalance: Number(productQuantity) - Number(quantity),
      description: "Transfer Delete",
      user: userId,
      date: new Date(),
    });
    await newTallyEntry.save();
  } catch (error) {
    console.error("Error creating tally card entry:", error);
    throw error;
  }
};

const createTallyCardEntryTransferOutDelete = async (
  productId,
  productQuantity,
  quantity,
  userId
) => {
  try {
    const newTallyEntry = new TallyCard({
      product: productId,
      inflow: Number(productQuantity) + Number(quantity),
      outflow: quantity,
      previousQty: Number(productQuantity),
      stockBalance: Number(productQuantity) + Number(quantity),
      description: "Transfer Delete",
      user: userId,
      date: new Date(),
    });
    await newTallyEntry.save();
  } catch (error) {
    console.error("Error creating tally card entry:", error);
    throw error;
  }
};

exports.removeTransfer = catchAsync(async (req, res, next) => {
  const transfer = await Transfer.findById(req.params.id);

  if (!transfer) {
    return next(new AppError("Transfer not found", 404));
  }

  await Transfer.findByIdAndDelete({
    _id: transfer._id,
  });

  const activity = new Activity({
    user: req.user._id,
    activityType: "remove_transfer",
    details: {
      transferId: transfer._id,
      amount: transfer.amount,
      quantity: transfer.quantity,
      products: transfer.products,
    },
  });
  await activity.save();
  res.status(200).json({ message: "transfer Deleted" });
});

exports.updateTransferDate = catchAsync(async (req, res, next) => {
  const { transferDate } = req.body;
  const { id } = req.params;
  // Find the transfer record by ID
  const transfer = await Transfer.findById(id);

  if (!transfer) {
    return res.status(404).json({ error: "transfer record not found" });
  }
  await Transfer.findByIdAndUpdate(
    id,
    {
      purchaseTime: transferDate,
    },
    {
      new: true,
    }
  );
  // Iterate over each product in the purchase record
  for (const product of transfer.products) {
    // Assuming each product has a transferId
    const productTransferId = product.transferId;

    // Find the product in the TallyCard schema by its transferId
    const tallyCardProduct = await TallyCard.findOne({
      transferId: productTransferId,
    });

    if (tallyCardProduct) {
      // Update the date and time of the product in the TallyCard schema
      tallyCardProduct.date = transferDate;
      await tallyCardProduct.save();
    } else {
      console.log(
        `Product with transferId ${productTransferId} not found in TallyCard.`
      );
      // Handle the case where the product is not found in the TallyCard schema
    }
  }

  res.status(200).json({ message: "TransferId date updated successfully" });
});

// Edit product count in the transfer document

exports.updateTransferQuantity = async (req, res) => {
  try {
    const { transferId, productId } = req.params; // Extracting transfer and product IDs
    let { count } = req.body; // Extracting the new count from request body

    // Convert count to number and validate
    count = Number(count);
    if (isNaN(count)) {
      return res.status(400).json({ message: "Count must be a valid number" });
    }

    // Find the transfer by ID and update the product count
    const transfer = await Transfer.findById(transferId);

    if (!transfer) {
      return res.status(404).json({ message: "Transfer record not found" });
    }
    const productFind = transfer.products.find(
      (product) => product._id.toString() === productId
    );
    // Find the index of the product in the array
    const productIndex = transfer.products.findIndex(
      (product) => product._id.toString() === productId
    );

    if (productIndex === -1) {
      return res.status(404).json({ message: "Product not found in transfer" });
    }
    const removeProductCount = await Product.findByIdAndUpdate(
      productId, // Match transfer and product
      { quantity: Number(productFind.quantity) - Number(productFind.count) }, // Update the count of the matching product
      { new: true } // Return updated document
    );
    // Ensure product count is treated as a number
    transfer.products[productIndex].count = count;

    // Calculate total count of all products, ensuring all values are numbers
    const totalCount = transfer.products.reduce(
      (sum, product) => sum + (Number(product.count) || 0),
      0
    );

    // Update transfer quantity with the calculated total count
    transfer.quantity = totalCount;

    // Save the updated document
    await transfer.save();

    // Update Product Count
    const updatedProductCount = await Transfer.findOneAndUpdate(
      { _id: transferId, "products._id": productId }, // Match transfer and product
      { $set: { "products.$.count": Number(count) } }, // Update the count of the matching product
      { new: true } // Return updated document
    );
    // return console.log(removeProductCount);
    // Update the count in the standalone Product collection
    const updatedProduct = await Product.findByIdAndUpdate(
      productId,
      { quantity: Number(count) + Number(removeProductCount.quantity) },
      { new: true }
    );

    res.status(200).json({
      message: "Success",
      totalQuantity: transfer.quantity,
      updatedProductCount,
      updatedProduct,
      transfer,
    });
  } catch (error) {
    res
      .status(500)
      .json({ message: "Internal Server Error", error: error.message });
  }
};
