const User = require("../models/userModel");
const Activity = require("../models/activityTrackerModel");
const AppError = require("../utils/appError");
const catchAsync = require("../utils/catchAsync");
const Route = require("../models/routeModel.js");
const mongoose = require("mongoose");
const Menu = require("../models/menuModel.js");
const { generate10DigitUUID } = require("../helpers/generater");
const { default: slugify } = require("slugify");
const { hashPassword } = require("../helpers/auth");

// add staff
exports.addUser = catchAsync(async (req, res, next) => {
  try {
    const { name, email, contactNum, role, routes = [], menus = [] } = req.body;

    // Log request body for debugging

    // Check if the user already exists
    const userExist = await User.findOne({ email }).exec();
    if (userExist) {
      return next(new AppError("Email is already taken", 400));
    }

    // Generate and hash password
    const password = generate10DigitUUID();
    const hashedPassword = await hashPassword(password);

    // Validate and filter menus and routes
    const validMenus = menus.filter((menu) =>
      mongoose.Types.ObjectId.isValid(menu)
    );
    const validRoutes = routes.filter((route) =>
      mongoose.Types.ObjectId.isValid(route)
    );

    if (routes.length !== validRoutes.length) {
      return res.status(404).json({ message: "Some routes not found" });
    }

    // Validate and fetch menus
    const menusFound = await Menu.find({ _id: { $in: validMenus } }).populate(
      "routes"
    );

    if (menusFound.length !== validMenus.length) {
      return res.status(404).json({ message: "Some menus not found" });
    }

    const combinedItems = [];
    const routeSet = new Set(); // To track added routes

    menusFound.forEach((menu) => {
      combinedItems.push({
        type: "menu",
        _id: menu._id,
        path: menu.path,
        name: menu.name,
      });
      menu.routes.forEach((route) => {
        if (!routeSet.has(route._id.toString())) {
          combinedItems.push({
            type: "submenu",
            _id: route._id,
            path: route.path,
            name: route.name,
            parentMenu: menu._id,
          });
          routeSet.add(route._id.toString());
        }
      });
    });

    // Create new user
    const user = new User({
      name,
      email,
      contactNum,
      username: slugify(name) + generate10DigitUUID(),
      role,
      menus: validMenus,
      assignedRoutes: validRoutes,
      password: hashedPassword,
      generatedPassword: password,
      combinedRoutes: combinedItems,
    });

    // Save user and populate role
    await user.save();
    await user.populate({
      path: "role",
      model: "Role",
      select: "name",
    });

    // Respond with the created user
    res.status(200).json(user);
  } catch (error) {
    console.error("Error in addUser:", error);
    next(error);
  }
});

// make user an admin
exports.makeUserAdmin = catchAsync(async (req, res, next) => {
  const user = await User.findById(req.params.id);
  if (!user) {
    return next(new AppError("User not found", 404));
  }

  await User.findByIdAndUpdate(
    user._id,
    {
      role: "admin",
    },
    { new: true }
  );
  res.status(200).send({ ok: true });
  // console.log(roleUpdated);
});
// make promoteUser
exports.promoteUser = catchAsync(async (req, res, next) => {
  // console.log(req.params.userId);
  // console.log(req.body);
  const user = await User.findById(req.params.userId);
  if (!user) {
    return next(new AppError("User not found", 404));
  }

  await User.findByIdAndUpdate(
    user._id,
    {
      role: req.params.userRole,
    },
    { new: true }
  );
  res.status(200).send({ ok: true });
  // console.log(roleUpdated);
});

//Reset user password
exports.regenerateUserPassword = catchAsync(async (req, res, next) => {
  const user = await User.findById(req.params.id);
  let password = generate10DigitUUID();
  let hashedPassword = await hashPassword(password);
  if (!user) {
    return next(new AppError("User not found", 404));
  }

  await User.findByIdAndUpdate(
    user._id,
    {
      password: hashedPassword,
      generatedPassword: password,
    },
    { new: true }
  );
  res.status(200).send({ ok: true });
});

// remove user as an admin
exports.removeAsAdmin = catchAsync(async (req, res, next) => {
  const user = await User.findById(req.params.id);
  if (!user) {
    return next(new AppError("User not found", 404));
  }
  await User.findByIdAndUpdate(
    user._id,
    {
      role: "seller",
    },
    { new: true }
  );
  res.status(200).send({ ok: true });
  // console.log(roleUpdated);
});

//  make user an active
exports.trashAndUnTrashUser = catchAsync(async (req, res, next) => {
  const { username } = req.params;
  const user = await User.findOne({ username }).select("+active");
  if (!user) {
    return next(new AppError("User not found", 404));
  }
  // console.log(user);
  await User.findOneAndUpdate(
    { username: user.username },
    {
      active: user.active ? false : true,
    },
    { new: true }
  );
  res.status(200).send({ ok: true });
  // console.log(roleUpdated);
});
//  make user an active
exports.isActiveCombineRoute = catchAsync(async (req, res, next) => {
  // const { username } = req.params;
  const user = await User.findOne({ username: req.user.username });
  if (!user) {
    return next(new AppError("User not found", 404));
  }
  const updated = await User.findOneAndUpdate(
    { username: user.username },
    {
      isActiveCombineRoute: false,
    },
    { new: true }
  );
  // console.log(updated);
  res.status(200).send({ ok: true });
  // console.log(roleUpdated);
});

// make un-active
exports.moveUserToTrash = catchAsync(async (req, res, next) => {
  const { username } = req.params;
  const user = await User.findOne({ username });
  if (!user) {
    return next(new AppError("User not found", 404));
  }
  await User.findOneAndUpdate(
    { username: user.username },
    {
      active: false,
    },
    { new: true }
  );
  res.status(200).send({ ok: true });
});

// get users
exports.getAllUsers = catchAsync(async (req, res) => {
  const users = await User.find({})
    .populate({
      path: "role",
      model: "Role",
      select: "name",
    })
    .populate({
      path: "assignedRoutes",
      model: "Route",
      select: "path name",
    })
    .populate({
      path: "menus",
      model: "Menu",
      populate: {
        path: "routes",
        model: "Route",
        select: "path name",
      },
    })
    .select("-password +active")
    .sort({ createdAt: -1 });

  res.status(200).send(users);
});

// delete user
exports.removeUser = catchAsync(async (req, res, next) => {
  const { userId } = req.params;
  const user = await User.findById({ _id: userId });
  if (!user) {
    return next(new AppError("User not found", 404));
  }

  await User.findByIdAndRemove(user._id);
  res.status(200).json({ message: "User Deleted" });
});

// get single user data
exports.readSingleUser = catchAsync(async (req, res, next) => {
  let { userId } = req.params;
  const user = await User.findById({ _id: userId })
    .select("-password")
    .populate({
      path: "role",
      model: "Role",
      populate: {
        path: "routes",
        model: "Route",
        select: "path name",
      },
    })
    .populate({
      path: "assignedRoutes",
      model: "Route",
      select: "path name",
    })
    .populate({
      path: "menus",
      model: "Menu",
      select: "path name",
    });

  if (!user) {
    return next(new AppError("User not found", 404));
  }
  res.status(200).send(user);
});
