// server.js
const { readdirSync } = require("fs");
require("dotenv").config();
const express = require("express");
const cookieParser = require("cookie-parser");
const connectDB = require("./config/dbConnect");
const cors = require("cors");
const socket = require("socket.io");
const bodyParser = require("body-parser");
const ErrorHandler = require("./utils/errors");
const {
  initializeDeletionFrequency,
  setupDailySummaryCron,
  setupProductCron,
  setupLoginHistoryDeletionCron,
  setupSalaryReminderCron,
} = require("./autoscripts");
const { registerSocketHandlers } = require("./socket/socketHandlers");

const allowedOrigins = [
  "https://edalowagrochemicals.com",
  "https://www.edalowagrochemicals.com",
  "https://aiyinasemain.edalowagrochemicals.com",
  "https://www.aiyinasemain.edalowagrochemicals.com",
  "http://localhost:5173",
  "http://localhost:5174",
];
const corsOptions = {
  credentials: true,
  origin: (origin, callback) => {
    if (!origin || allowedOrigins.includes(origin)) {
      callback(null, true);
    } else {
      callback(new Error("Not allowed by CORS"));
    }
  },
  methods: ["GET", "POST", "PUT", "PATCH", "DELETE", "OPTIONS"],
  allowedHeaders: [
    "Origin",
    "X-Requested-With",
    "Content-Type",
    "Accept",
    "Authorization",
  ],
};

(async () => {
  try {
    await connectDB();
    // ✅ Configure web push after DB connection and before anything else
    const app = express();

    app.use(cors(corsOptions));
    app.use(express.json({ limit: "30mb" }));
    app.use(cookieParser());
    app.use(bodyParser.urlencoded({ extended: true }));
    app.options("*", cors(corsOptions));

    readdirSync("./routes").map((r) =>
      app.use("/api", require(`./routes/${r}`))
    );

    app.get("/", (req, res) => {
      res.send("<h4>POS SERVER</h4>");
    });

    app.use(ErrorHandler);

    const PORT = process.env.PORT || 8000;
    const server = app.listen(PORT, () => {
      console.log(`Server running on port ::: ${PORT}`);
      try {
        initializeDeletionFrequency();
        setupDailySummaryCron();
        setupSalaryReminderCron();
        // makeProductActive();
        // setupLoginHistoryDeletionCron();
      } catch (e) {
        console.error("Auto script init error:", e.message);
      }
    });

    // SOCKET.IO INTEGRATION
    const io = socket(server, {
      cors: {
        origin: allowedOrigins, // Using the allowedOrigins array here
        credentials: true,
      },
    });

    global.onlineUsers = new Map();
    // Register all socket event handlers
    registerSocketHandlers(io);
  } catch (err) {
    console.error("Startup failed:", err.message);
    process.exit(1);
  }
})();
