// middleware/authMiddleware.js
const jwt = require("jsonwebtoken");
const User = require("../models/userModel");

exports.verifyTokenOld = async (req, res, next) => {
  const authHeader = req.header("Authorization");
  if (!authHeader || !authHeader.startsWith("Bearer")) {
    return res.status(401).json({ message: "Access Denied" });
  }

  const token = authHeader.replace("Bearer", "");
  if (!token) {
    return res.status(401).json({ message: "Access Denied" });
  }

  try {
    const decoded = jwt.verify(token, process.env.JWT_ACCESS_SECRET);
    req.user = decoded;
    const user = await User.findById(req.user.id).populate("role");
    if (!user) {
      return res.status(401).json({ message: "User not found" });
    }
    // Optionally, check for role permissions or other conditions here
    req.user.role = user.role;
    next();
  } catch (err) {
    res.status(400).json({ message: "Invalid Token" });
  }
};

exports.verifyTokenOlder = async (req, res, next) => {
  let token;

  // Check Authorization header
  const authHeader = req.header("Authorization");
  if (authHeader && authHeader.startsWith("Bearer ")) {
    token = authHeader.replace("Bearer ", "");
  }

  // If token isn't in header, check cookies
  if (!token && req.cookies) {
    token = req.cookies.refreshToken;
  }

  if (!token) {
    return res.status(401).json({ message: "Access Denied" });
  }

  try {
    // Verify token
    const decoded = jwt.verify(token, process.env.JWT_ACCESS_SECRET);
    req.user = decoded;
    // Find user and populate role
    const user = await User.findById(req.user.id).populate("role");
    if (!user) {
      return res.status(401).json({ message: "User not found" });
    }
    // Attach the role to the request
    req.user.role = user.role;
    // Proceed to the next middleware
    next();
  } catch (err) {
    if (err.name === "TokenExpiredError") {
      return res.status(401).json({ message: "Token expired" });
    }
    res.status(400).json({ message: "Invalid Token" });
  }
};

exports.verifyToken = async (req, res, next) => {
  let token;

  // Check if authorization header is present and starts with Bearer
  if (
    req.headers.authorization &&
    req.headers.authorization.startsWith("Bearer")
  ) {
    try {
      // Extract token from the Authorization header
      token = req.headers.authorization.split(" ")[1];

      // Verify the token using the JWT_ACCESS_SECRET
      const decoded = jwt.verify(token, process.env.JWT_ACCESS_SECRET);

      // Find the user by the decoded ID and populate the role
      req.user = await User.findById(decoded.id).populate("role");

      // If the user is not found, return an error
      if (!req.user) {
        return res.status(401).json({
          flag: false,
          error: "User not found",
        });
      }
      // Optionally, you can check role permissions or other user conditions here
      // req.user.role = user.role;

      next(); // Proceed to the next middleware or route handler
    } catch (err) {
      // Handle any error during token verification
      return res.status(401).json({
        flag: false,
        error: "Not authorized, Token is not valid",
      });
    }
  }

  // If no token was provided in the authorization header
  if (!token) {
    return res.status(401).json({
      flag: false,
      error: "Not authorized, No token provided",
    });
  }
};

exports.verifyRouteAccess = (route) => {
  return (req, res, next) => {
    if (!req.user) {
      return res.status(401).json({ message: "Unauthorized" });
    }

    // Ensure req.user.role.routes and req.user.assignedRoutes are arrays
    const roleRoutes =
      req.user.role && req.user.role.routes ? req.user.role.routes : [];
    const assignedRoutes = req.user.assignedRoutes
      ? req.user.assignedRoutes
      : [];
    // console.log(roleRoutes);

    // Combine role routes and assigned routes for easier checking
    const allRoutes = [...roleRoutes, ...assignedRoutes];
    // Check if the route path exists in the user's accessible routes
    const hasAccess = roleRoutes.some((r) => r.path === route);

    if (!hasAccess) {
      return res.status(403).json({ status: false, message: "Access Denied" });
    }

    next();
  };
};
