// utils/notificationService.js
const PushNotifications = require("node-pushnotifications");
const User = require("../models/userModel");
const PushSubscription = require("../models/pushSubscriptionModel");

// Initialize push notifications service
const push = new PushNotifications({
  web: {
    vapidDetails: {
      subject: `mailto:${process.env.VAPID_EMAIL}`,
      publicKey: process.env.VAPID_PUBLIC_KEY,
      privateKey: process.env.VAPID_PRIVATE_KEY,
    },
    gcmAPIKey: process.env.GCM_API_KEY, // Optional for Android
    TTL: 2419200, // 4 weeks
    contentEncoding: "aes128gcm", // or "aesgcm"
  },
});

/**
 * Send push notification to user
 */
exports.sendPushNotification = async (userId, payload) => {
  try {
    // Get user with subscriptions
    const user = await User.findById(userId)
      .populate("pushSubscriptions")
      .lean();
    if (!user) {
      console.error(`User ${userId} not found`);
      return { success: false, message: "User not found" };
    }

    if (!user.pushSubscriptions || user.pushSubscriptions.length === 0) {
      console.warn(`No push subscriptions found for user ${userId}`);
      return { success: false, message: "No active subscriptions" };
    }

    // Prepare notification data
    const notificationData = {
      title: payload.title || "New Notification",
      body: payload.body || "You have a new notification",
      icon:
        payload.icon ||
        "https://cdn2.vectorstock.com/i/thumb-large/94/66/emoji-smile-icon-symbol-smiley-face-vector-26119466.jpg",
      data: {
        url: payload.url || "/notifications",
        userId: userId.toString(),
        ...payload.data,
      },
      actions: payload.actions || [],
      vibrate: payload.vibrate || [200, 100, 200],
    };

    // Convert subscriptions to format expected by node-pushnotifications
    const registrations = user.pushSubscriptions.map((sub) => ({
      endpoint: sub.endpoint,
      keys: sub.keys,
    }));

    // Send notifications
    const results = await push.send(registrations, notificationData);

    // Process results
    const summary = {
      total: registrations.length,
      successful: 0,
      failed: 0,
      removed: 0,
      errors: [],
    };

    results.forEach((result, index) => {
      if (result.success) {
        summary.successful++;
      } else {
        summary.failed++;
        summary.errors.push({
          subscriptionId: user.pushSubscriptions[index]._id,
          error: result.message,
        });

        // Check for expired/invalid subscriptions
        if (
          result.error &&
          (result.error.statusCode === 410 || result.error.statusCode === 404)
        ) {
          cleanupInvalidSubscription(userId, user.pushSubscriptions[index]._id);
          summary.removed++;
        }
      }
    });

    return { success: summary.failed === 0, ...summary };
  } catch (error) {
    console.error("Error in sendPushNotification:", error);
    return { success: false, error: error.message };
  }
};

/**
 * Clean up invalid subscriptions
 */
async function cleanupInvalidSubscription(userId, subscriptionId) {
  try {
    await Promise.all([
      User.findByIdAndUpdate(userId, {
        $pull: { pushSubscriptions: subscriptionId },
      }),
      PushSubscription.findByIdAndDelete(subscriptionId),
    ]);
    console.log(`Cleaned up invalid subscription ${subscriptionId}`);
  } catch (error) {
    console.error(`Error cleaning up subscription ${subscriptionId}:`, error);
  }
}

/**
 * Subscribe user to push notifications
 */
exports.subscribeUser = async (userId, subscriptionData) => {
  try {
    // Check if subscription already exists
    const existingSub = await PushSubscription.findOne({
      endpoint: subscriptionData.endpoint,
    });

    if (existingSub) {
      return { success: true, subscription: existingSub };
    }

    // Create new subscription
    const newSubscription = new PushSubscription({
      endpoint: subscriptionData.endpoint,
      keys: subscriptionData.keys,
      user: userId,
    });

    await newSubscription.save();

    // Add to user's subscriptions
    await User.findByIdAndUpdate(userId, {
      $addToSet: { pushSubscriptions: newSubscription._id },
    });

    return { success: true, subscription: newSubscription };
  } catch (error) {
    console.error("Error subscribing user:", error);
    return { success: false, error: error.message };
  }
};
