const User = require("../models/userModel");
const Menu = require("../models/menuModel.js");
const Permission = require("../models/permissionModel");
const Route = require("../models/routeModel.js");
const Role = require("../models/roleModel.js");
const { generateTokens } = require("../utils/token.js");
const Notification = require("../models/notificationModel");

const Cart = require("../models/cartModel");
const { generateUniqueCode } = require("../helpers/generateUniqueCode.js");

exports.registerSocketHandlers = (io) => {
  io.on("connection", (socket) => {
    global.chatSocket = socket;
    // Join the room for the user
    socket.on("join-room", (userId) => {
      socket.join(userId);
    });
    socket.on("join-notifications-room", (userId) => {
      socket.join(userId);
      // Immediately send current notifications upon joining
      fetchAndSendNotifications(userId, socket);
    });
    // 📌 Get user permissions
    socket.on("get-user-permissions", async (userId) => {
      try {
        if (!userId) {
          return socket.emit("user-permissions-error", {
            message: "User ID is required",
          });
        }

        const permissions = await Permission.findOne({ user: userId });
        if (!permissions) {
          return socket.emit("user-permissions-error", {
            message: "No permissions assigned to this user",
          });
        }

        socket.emit("user-permissions-success", {
          message: "Permissions fetched successfully",
          permissions,
        });
      } catch (err) {
        console.error("Socket GetUserPermissions Error:", err.message);
        socket.emit("user-permissions-error", {
          message: "Internal server error",
        });
      }
    });

    // 📌 Get loginuser permissions
    socket.on("get-loginuser-permissions", async (userId) => {
      try {
        if (!userId) {
          return socket.emit("loginuser-permissions-error", {
            message: "User ID is required",
          });
        }

        const permissions = await Permission.findOne({ user: userId });
        if (!permissions) {
          return socket.emit("loginuser-permissions-error", {
            message: "No permissions assigned to this user",
          });
        }

        socket.emit("loginuser-permissions-success", {
          message: "Permissions fetched successfully",
          permissions,
        });
      } catch (err) {
        console.error("Socket GetUserPermissions Error:", err.message);
        socket.emit("loginuser-permissions-error", {
          message: "Internal server error",
        });
      }
    });

    // 📌 Assign user permissions
    socket.on(
      "assign-user-permissions",
      async ({ userId, permissions }, callback) => {
        try {
          const updated = await Permission.findOneAndUpdate(
            { user: userId },
            permissions,
            { new: true, upsert: true }
          );

          // Broadcast the update to all connected clients
          io.emit("permissions-updated", {
            userId,
            permissions: updated,
          });
          socket.emit("user-permissions-success", { permissions: updated });
          if (callback) callback({ success: true });
        } catch (err) {
          socket.emit("user-permissions-error", { message: err.message });
          if (callback) callback({ error: err.message });
        }
      }
    );
    // Get All users
    socket.on("get-all-users", async () => {
      try {
        // Only return non-sensitive fields
        const users = await User.find({})
          .populate({
            path: "role",
            model: "Role",
            select: "name",
          })
          .populate({
            path: "assignedRoutes",
            model: "Route",
            select: "path name",
          })
          .populate({
            path: "menus",
            model: "Menu",
            populate: {
              path: "routes",
              model: "Route",
              select: "path name",
            },
          })
          .select("-password +active")
          .sort({ createdAt: -1 });
        socket.emit("all-users-success", users);
      } catch (err) {
        console.error("Get Users Error:", err);
        socket.emit("all-users-error", {
          message: err.message || "Internal server error",
        });
      }
    });
    // Get All Menus
    socket.on("get-all-menus", async () => {
      try {
        // Only return non-sensitive fields
        const menus = await Menu.find({}).populate({
          path: "routes",
          model: "Route",
          select: "path name",
        });
        socket.emit("all-menus-success", menus);
      } catch (err) {
        console.error("Get Menus Error:", err);
        socket.emit("all-menus-error", {
          message: err.message || "Internal server error",
        });
      }
    });
    // Get All Routes
    socket.on("get-all-routes", async () => {
      try {
        // Only return non-sensitive fields
        const routes = await Route.find({}).sort({ name: 1 });
        socket.emit("all-routes-success", routes);
      } catch (err) {
        console.error("Get Routes Error:", err);
        socket.emit("all-routes-error", {
          message: err.message || "Internal server error",
        });
      }
    });

    // Get single user details
    socket.on("get-user-details", async (userId) => {
      try {
        if (!userId) {
          return socket.emit("user-details-error", {
            message: "User ID is required",
          });
        }
        const user = await User.findById({ _id: userId })
          .select("-password")
          .populate({
            path: "role",
            model: "Role",
            populate: {
              path: "routes",
              model: "Route",
              select: "path name",
            },
          })
          .populate({
            path: "assignedRoutes",
            model: "Route",
            select: "path name",
          })
          .populate({
            path: "menus",
            model: "Menu",
            select: "path name",
          });

        if (!user) {
          return socket.emit("user-details-error", {
            message: "No details assigned to this user",
          });
        }
        socket.emit("user-details-success", user);
      } catch (err) {
        console.error("Socket GetUserDetails Error:", err.message);
        socket.emit("user-details-error", {
          message: "Internal server error",
        });
      }
    });

    // 📌 Assign roles route and menus
    socket.on(
      "assign-role-route-menus",
      async ({ userId, roleId, routeId, menuId }, callback) => {
        try {
          const user = await User.findById(userId);
          const role = await Role.findById(roleId).populate("routes");

          // Validate and fetch
          const routes = await Route.find({ _id: { $in: routeId } });
          const menus = await Menu.find({ _id: { $in: menuId } }).populate(
            "routes"
          );

          if (
            !user ||
            !role ||
            routes.length !== routeId.length ||
            menus.length !== menuId.length
          ) {
            const errorMsg = "User, role, or some routes/menus not found";
            if (callback) callback({ error: errorMsg });
            return;
          }

          // Update role routes if needed
          routes.forEach((route) => {
            if (!role.routes.some((r) => r._id.equals(route._id))) {
              role.routes.push(route._id);
            }
          });
          await role.save();

          // Assign role and update menus/routes
          user.role = role._id;
          user.assignedRoutes = routeId;
          user.menus = menuId;

          // Combine routes
          const newCombinedRoutes = [];
          const addToCombinedRoutes = (item, type, parentMenu = null) => {
            if (!newCombinedRoutes.some((r) => r._id.equals(item._id))) {
              newCombinedRoutes.push({
                ...item.toObject(),
                type,
                parentMenu,
              });
            }
          };

          routes.forEach((route) => addToCombinedRoutes(route, "route"));
          menus.forEach((menu) => {
            addToCombinedRoutes(menu, "menu");
            menu.routes.forEach((route) =>
              addToCombinedRoutes(route, "submenu", menu._id)
            );
          });

          user.combinedRoutes = newCombinedRoutes;
          user.isActiveCombineRoute = true;
          await user.save();

          // Emit updated routes in real-time
          io.emit("user-combined-routes-updated", {
            userId: user._id.toString(),
            combinedRoutes: newCombinedRoutes,
          });

          if (callback) callback({ success: true });
        } catch (err) {
          console.error("assign-role-route-menus error:", err);
          if (callback) callback({ error: err.message });
        }
      }
    );

    socket.on("get-user-combined-routes", async ({ userId }, callback) => {
      try {
        const currentUser = await User.findById(userId);
        if (!currentUser) {
          if (callback) callback({ error: "User not found" });
          return;
        }

        // Reorder combinedRoutes: routes first
        const combinedRoutes = currentUser.combinedRoutes.sort((a, b) => {
          if (a.type === "route" && b.type !== "route") return -1;
          if (a.type !== "route" && b.type === "route") return 1;
          return 0;
        });

        // console.log("from get-user-combined-routes", combinedRoutes);

        if (callback) callback({ success: true, combinedRoutes });
      } catch (error) {
        console.error("Error retrieving user combined routes:", error);
        if (callback) callback({ error: "Server error: " + error.message });
      }
    });

    socket.on("add-to-cart", async ({ userId, products }) => {
      try {
        products = Array.isArray(products) ? products : [products];

        if (!products.length || !products.every((p) => p._id)) {
          return socket.emit("cart-error", {
            success: false,
            message:
              "Invalid product data format. Each product must have an _id",
          });
        }

        if (!userId) {
          return socket.emit("cart-error", {
            success: false,
            message: "User ID is required for cart operations",
          });
        }

        let cart = await Cart.findOne({ user: userId });
        if (!cart) {
          cart = new Cart({ user: userId, items: [] });
        }

        for (const product of products) {
          const productId = product._id.toString();
          const existingItem = cart.items.find(
            (item) => item._id.toString() === productId
          );

          if (existingItem) {
            existingItem.count =
              (existingItem.count || 1) + (product.count || 1);
            existingItem.updatedAt = new Date();
          } else {
            cart.items.push({
              ...product,
              _id: product._id,
              count: product.count || 1,
              // addedAt: new Date(),
              // updatedAt: new Date(),
              salesId: generateUniqueCode(),
            });
          }
        }

        cart.markModified("items");
        await cart.save();

        const totalItems = cart.items.reduce(
          (sum, item) => sum + (item.count || 1),
          0
        );
        const subtotal = cart.items.reduce(
          (sum, item) => sum + (item.count || 1) * (item.sellingPrice || 0),
          0
        );

        const formattedItems = cart.items.map((item) => ({
          _id: item._id,
          name: item.name,
          count: item.count,
          sellingPrice: item.sellingPrice,
          salesId: item.salesId,
          // addedAt: item.addedAt,
          // updatedAt: item.updatedAt,
        }));

        // ✅ Emit real-time update to the user's room
        io.to(userId).emit("cart-updated", {
          success: true,
          totalItems,
          subtotal,
          items: formattedItems,
        });

        // ✅ Confirm success to the sender as well
        socket.emit("cart-add-success", {
          success: true,
          message: "Product(s) added to cart",
        });
      } catch (error) {
        console.error("Socket AddToCart Error:", error.message);
        socket.emit("cart-error", {
          success: false,
          message: "Server error while updating cart",
          error: error.message,
        });
      }
    });

    // 📌 Get user's cart items in real-time
    socket.on("get-user-cart", async (userId) => {
      try {
        if (!userId) {
          return socket.emit("user-cart-error", {
            message: "User ID is required",
          });
        }

        const cart = await Cart.findOne({
          user: userId,
          status: "active",
        }).populate("user", "name email");

        if (!cart) {
          return io.to(userId).emit("cart-updated", {
            success: true,
            message: "No active cart found",
            items: [],
            totalItems: 0,
            subtotal: 0,
          });
        }

        const totalItems = cart.items.reduce(
          (sum, item) => sum + (item.count || 0),
          0
        );
        const subtotal = cart.items.reduce(
          (sum, item) => sum + (item.count || 0) * (item.sellingPrice || 0),
          0
        );

        const formattedItems = cart.items.map((item) => ({
          _id: item._id,
          name: item.name,
          count: item.count,
          sellingPrice: item.sellingPrice,
          costPrice: item.costPrice,
          quantity: item.quantity,
          expireDate: item.expireDate,
          category: item.category,
          salesId: item.salesId,
          // addedAt: item.addedAt,
          // updatedAt: item.updatedAt,
        }));

        // Emit to user room
        io.to(userId).emit("cart-updated", {
          success: true,
          totalItems,
          subtotal,
          items: formattedItems,
          createdAt: cart.createdAt,
          updatedAt: cart.updatedAt,
        });

        // Optionally also emit the original event response
        socket.emit("user-cart-success", {
          success: true,
          cartId: cart._id,
          userId: cart.user._id,
          totalItems,
          subtotal,
          items: formattedItems,
          createdAt: cart.createdAt,
          updatedAt: cart.updatedAt,
        });
      } catch (error) {
        console.error("Socket GetUserCart Error:", error.message);
        socket.emit("user-cart-error", {
          success: false,
          message: "Server error while fetching cart items",
          error: error.message,
        });
      }
    });
    socket.on("remove-from-cart", async ({ userId, productId }, callback) => {
      try {
        if (!userId || !productId) {
          const errorMsg = "Both userId and productId are required";
          socket.emit("cart-error", { success: false, message: errorMsg });
          if (callback) callback({ error: errorMsg });
          return;
        }

        const cart = await Cart.findOne({ user: userId });
        if (!cart) {
          const errorMsg = "Cart not found for this user";
          socket.emit("cart-error", { success: false, message: errorMsg });
          if (callback) callback({ error: errorMsg });
          return;
        }

        const originalLength = cart.items.length;
        cart.items = cart.items.filter(
          (item) => item._id.toString() !== productId.toString()
        );

        if (cart.items.length === originalLength) {
          const warningMsg = "Item not found in cart";
          socket.emit("cart-warning", { success: false, message: warningMsg });
          if (callback) callback({ warning: warningMsg });
          return;
        }

        cart.markModified("items");
        await cart.save();

        const totalItems = cart.items.reduce(
          (sum, item) => sum + (item.count || 1),
          0
        );
        const subtotal = cart.items.reduce(
          (sum, item) => sum + (item.count || 1) * (item.sellingPrice || 0),
          0
        );

        const formattedItems = cart.items.map((item) => ({
          _id: item._id,
          name: item.name,
          count: item.count,
          sellingPrice: item.sellingPrice,
          salesId: item.salesId,
          // addedAt: item.addedAt,
          // updatedAt: item.updatedAt,
        }));

        socket.emit("cart-updated", {
          items: formattedItems,
          totalItems,
          subtotal,
        });

        if (callback) callback({ success: true });
      } catch (err) {
        console.error("Remove from cart error:", err.message);
        socket.emit("cart-error", {
          success: false,
          message: "Server error: " + err.message,
        });
        if (callback) callback({ error: err.message });
      }
    });
    socket.on("update-cart-item", async ({ productId, count, userId }) => {
      try {
        // Update the product count in the user's cart in your DB
        const cart = await Cart.findOneAndUpdate(
          { user: userId, "items._id": productId },
          { $set: { "items.$.count": count } },
          { new: true }
        );

        const totalItems = cart.items.reduce(
          (sum, item) => sum + (item.count || 1),
          0
        );
        const subtotal = cart.items.reduce(
          (sum, item) => sum + (item.count || 1) * (item.sellingPrice || 0),
          0
        );
        // console.log(cart);
        if (cart) {
          io.to(userId).emit("cart-updated", {
            success: true,
            totalItems,
            subtotal,
            items: cart.items,
          });
        }
      } catch (err) {
        console.error("Error updating cart item:", err);
        socket.emit("user-cart-error", {
          success: false,
          message: "Update failed",
        });
      }
    });
    // In your backend socket implementation
    socket.on("clear-user-cart", async (userId) => {
      try {
        await Cart.findOneAndUpdate(
          { user: userId },
          { $set: { items: [] } },
          { new: true }
        );
        socket.emit("cart-updated", {
          success: true,
          items: [],
          totalItems: 0,
        });
      } catch (error) {
        socket.emit("cart-error", { message: "Failed to clear cart" });
      }
    });

    // 📌 Get user's notifications items in real-time
    // socket.on("get-user-notifications", async (userId) => {
    //   try {
    //     if (!userId) {
    //       return socket.emit("user-notifications-error", {
    //         message: "User ID is required",
    //       });
    //     }

    //     const notifications = await Notification.find({ user: userId });
    //     const unreadCount = await Notification.countDocuments({
    //       user: userId,
    //       read: false,
    //     });

    //     if (!notifications) {
    //       return socket.emit("user-notifications-error", {
    //         message: "No notifications assigned to this user",
    //       });
    //     }
    //     socket.emit("user-notifications-success", {
    //       notifications,
    //       unreadCount,
    //     });
    //   } catch (err) {
    //     console.error("Socket Get User Notifications Error:", err.message);
    //     socket.emit("user-notifications-error", {
    //       message: "Internal server error",
    //     });
    //   }
    // });

    // Server-side socket implementation should match these events:
    socket.on("get-user-notifications", async (userId, callback) => {
      try {
        await fetchAndSendNotifications(userId, socket);
        if (callback) callback({ success: true });
      } catch (error) {
        if (callback) callback({ success: false, error: error.message });
      }
    });

    // Mark notification as read
    socket.on(
      "mark-notification-read",
      async ({ notificationId, userId }, callback) => {
        try {
          await Notification.findByIdAndUpdate(notificationId, { read: true });
          const unreadCount = await Notification.countDocuments({
            user: userId,
            read: false,
          });

          // Emit both the full update and just the count update
          await fetchAndSendNotifications(userId, socket);
          socket
            .to(userId)
            .emit("unread-count-updated", { count: unreadCount });

          if (callback) callback({ success: true });
        } catch (error) {
          if (callback) callback({ success: false, error: error.message });
        }
      }
    );

    // Delete notification
    socket.on(
      "delete-notification",
      async ({ notificationId, userId }, callback) => {
        try {
          // Get notification first to check if it was unread
          const notification = await Notification.findById(notificationId);
          if (!notification) {
            throw new Error("Notification not found");
          }

          const wasUnread = !notification.read;
          await Notification.findByIdAndDelete(notificationId);

          const unreadCount = wasUnread
            ? await Notification.countDocuments({ user: userId, read: false })
            : undefined;

          // Emit updates
          await fetchAndSendNotifications(userId, socket);
          if (unreadCount !== undefined) {
            socket
              .to(userId)
              .emit("unread-count-updated", { count: unreadCount });
          }

          if (callback)
            callback({
              success: true,
              unreadCount,
            });
        } catch (error) {
          if (callback) callback({ success: false, error: error.message });
        }
      }
    );

    // Helper function to fetch and broadcast notifications
    const fetchAndSendNotificationsOld = async (userId, socket) => {
      const notifications = await Notification.find({ user: userId }).sort({
        createdAt: -1,
      });
      // console.log(notifications);
      const unreadCount = await Notification.countDocuments({
        user: userId,
        read: false,
      });

      // Send full notifications update
      io.to(userId).emit("notifications-updated", {
        notifications,
        unreadCount,
      });

      // Also send just the count update
      io.to(userId).emit("unread-count-updated", {
        count: unreadCount,
      });
    };
    // 🔄 Global helper to fetch and send all notifications
    const fetchAndSendNotifications = async (userId, socket) => {
      const notifications = await Notification.find({ user: userId }).sort({
        createdAt: -1,
      });

      const unreadCount = await Notification.countDocuments({
        user: userId,
        read: false,
      });

      // Send to specific socket if available
      if (socket) {
        socket.emit("notifications-updated", {
          notifications,
          unreadCount,
        });
      }

      // Send to user's room for real-time updates
      io.to(userId).emit("notifications-updated", {
        notifications,
        unreadCount,
      });

      io.to(userId).emit("unread-count-updated", {
        count: unreadCount,
      });
    };

    // ✅ Helper to send a single real-time notification
  });
};

// socket.on("get-user-permissions", async (userId) => {
//   try {
//     if (!userId) {
//       return socket.emit("user-permissions-error", {
//         message: "User ID is required",
//       });
//     }

//     const permissions = await Permission.findOne({ user: userId });
//     if (!permissions) {
//       return socket.emit("user-permissions-error", {
//         message: "No permissions assigned to this user",
//       });
//     }

//     socket.emit("user-permissions-success", {
//       message: "Permissions fetched successfully",
//       permissions,
//     });
//   } catch (err) {
//     console.error("Socket GetUserPermissions Error:", err.message);
//     socket.emit("user-permissions-error", {
//       message: "Internal server error",
//     });
//   }
// });

// Get user notifications
